// Background Service Worker
// Handles extension lifecycle, messaging, and API coordination

import api from '../lib/api.js';
import Storage from '../lib/storage.js';

// Initialize API on startup
api.init().then(isAuth => {
    console.log('exploit.team AutoJob initialized, authenticated:', isAuth);
});

// Message handler for content scripts and popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    handleMessage(request, sender).then(sendResponse);
    return true; // Keep channel open for async response
});

async function handleMessage(request, sender) {
    const { action, data } = request;

    try {
        switch (action) {
            // Authentication
            case 'login':
                return await api.login(data.username, data.password);

            case 'logout':
                return await api.logout();

            case 'checkAuth':
                await api.init();
                return {
                    success: true,
                    isAuthenticated: api.isAuthenticated(),
                    user: api.getUser()
                };

            // Generation
            case 'generate':
                // Check daily limit
                const todayCount = await Storage.getTodayCount();
                const settings = await Storage.getSettings();

                if (todayCount >= settings.dailyLimit) {
                    return {
                        success: false,
                        error: `Daily limit reached (${settings.dailyLimit} applications). Try again tomorrow.`
                    };
                }

                // Check blacklist
                if (await Storage.isBlacklisted(data.company)) {
                    return {
                        success: false,
                        error: `${data.company} is in your blacklist.`
                    };
                }

                // Check for duplicate
                if (await Storage.hasApplied(data.jobUrl)) {
                    return {
                        success: false,
                        error: 'You have already applied to this job.'
                    };
                }

                return await api.generate(data);

            // Generate custom answer for any question
            case 'generateCustomAnswer':
                try {
                    const answerResponse = await api.generateCustomAnswer(
                        data.question,
                        data.jobTitle,
                        data.company,
                        data.jobDescription
                    );
                    return answerResponse;
                } catch (error) {
                    return { success: false, error: error.message };
                }

            // Document download
            case 'download':
                const blob = await api.downloadDocument(
                    data.type,
                    data.content,
                    data.format,
                    data.style
                );

                // Convert blob to base64 for transfer
                const reader = new FileReader();
                return new Promise(resolve => {
                    reader.onloadend = () => {
                        resolve({
                            success: true,
                            data: reader.result,
                            filename: `${data.type}-${Date.now()}.${data.format}`
                        });
                    };
                    reader.readAsDataURL(blob);
                });

            // Resume
            case 'getResume':
                return await api.getResume();

            // Application tracking
            case 'logApplication':
                const app = await Storage.addApplication(data);

                // Show notification
                const notifySettings = await Storage.getSettings();
                if (notifySettings.showNotifications) {
                    chrome.notifications.create({
                        type: 'basic',
                        iconUrl: 'assets/icons/icon128.png',
                        title: 'Application Submitted',
                        message: `Applied to ${data.title} at ${data.company}`
                    });
                }

                return { success: true, application: app };

            case 'getApplicationHistory':
                const history = await Storage.getApplicationHistory();
                return { success: true, history };

            case 'getTodayCount':
                const count = await Storage.getTodayCount();
                return { success: true, count };

            // Settings
            case 'getSettings':
                const currentSettings = await Storage.getSettings();
                return { success: true, settings: currentSettings };

            case 'updateSettings':
                await Storage.updateSettings(data);
                return { success: true };

            // Blacklist
            case 'addToBlacklist':
                await Storage.addToBlacklist(data.company);
                return { success: true };

            case 'checkBlacklist':
                const isBlacklisted = await Storage.isBlacklisted(data.company);
                return { success: true, isBlacklisted };

            // User profile for auto-fill
            case 'getUserProfile':
                const profileResult = await chrome.storage.local.get('userProfile');
                return { success: true, profile: profileResult.userProfile || null };

            case 'saveUserProfile':
                await chrome.storage.local.set({ userProfile: data });
                return { success: true };

            case 'updateUserProfile':
                const currentProfile = await chrome.storage.local.get('userProfile');
                const mergedProfile = { ...(currentProfile.userProfile || {}), ...data };
                await chrome.storage.local.set({ userProfile: mergedProfile });
                return { success: true, profile: mergedProfile };

            // Auto-apply workflow
            case 'autoApply':
                // This triggers the full auto-apply in the content script
                // Background just validates and returns config
                const autoApplySettings = await Storage.getSettings();
                return {
                    success: true,
                    settings: autoApplySettings,
                    autoSubmit: data.autoSubmit || false
                };

            default:
                return { success: false, error: `Unknown action: ${action}` };
        }
    } catch (error) {
        console.error('Message handler error:', error);
        return { success: false, error: error.message };
    }
}

// Handle extension install/update
chrome.runtime.onInstalled.addListener(async (details) => {
    if (details.reason === 'install') {
        console.log('exploit.team AutoJob installed!');

        // Set default settings
        await Storage.updateSettings({});

        // Open options page for initial setup
        chrome.runtime.openOptionsPage();
    } else if (details.reason === 'update') {
        console.log('exploit.team AutoJob updated to', chrome.runtime.getManifest().version);
    }
});

// Badge management - show application count
async function updateBadge() {
    const count = await Storage.getTodayCount();
    const settings = await Storage.getSettings();

    if (count > 0) {
        chrome.action.setBadgeText({ text: count.toString() });
        chrome.action.setBadgeBackgroundColor({
            color: count >= settings.dailyLimit ? '#ef4444' : '#22c55e'
        });
    } else {
        chrome.action.setBadgeText({ text: '' });
    }
}

// Update badge periodically
setInterval(updateBadge, 60000);
updateBadge();
