#!/bin/bash
# Eric's AutoApply Extension - Build and Publish Script
# Automatically increments version, builds zip, and updates server manifest

set -e

EXTENSION_DIR="/var/www/job/extension"
PUBLIC_DIR="/var/www/job/public"
MANIFEST_FILE="$EXTENSION_DIR/manifest.json"
UPDATE_FILE="$PUBLIC_DIR/extension-update.json"
ZIP_FILE="$PUBLIC_DIR/jobgen-pro-extension.zip"

# Get current version from manifest
CURRENT_VERSION=$(grep -o '"version": "[^"]*"' "$MANIFEST_FILE" | cut -d'"' -f4)
echo "Current version: $CURRENT_VERSION"

# Parse version parts
IFS='.' read -ra VERSION_PARTS <<< "$CURRENT_VERSION"
MAJOR=${VERSION_PARTS[0]:-2}
MINOR=${VERSION_PARTS[1]:-0}
PATCH=${VERSION_PARTS[2]:-0}

# Increment patch version
NEW_PATCH=$((PATCH + 1))
NEW_VERSION="$MAJOR.$MINOR.$NEW_PATCH"
echo "New version: $NEW_VERSION"

# Update manifest.json with new version
sed -i "s/\"version\": \"$CURRENT_VERSION\"/\"version\": \"$NEW_VERSION\"/" "$MANIFEST_FILE"
echo "Updated manifest.json"

# Get changelog from command line or use default
CHANGELOG="${1:-Bug fixes and improvements}"

# Build the zip
cd "$EXTENSION_DIR"
rm -f "$ZIP_FILE"
zip -r "$ZIP_FILE" . -x "*.git*" -x "*node_modules*" -x "*.DS_Store"
echo "Built zip: $ZIP_FILE"

# Update the server manifest
cat > "$UPDATE_FILE" << EOF
{
  "version": "$NEW_VERSION",
  "download_url": "https://job.exploit.bot/jobgen-pro-extension.zip",
  "changelog": "$CHANGELOG",
  "updated_at": "$(date -u +"%Y-%m-%dT%H:%M:%SZ")"
}
EOF
echo "Updated $UPDATE_FILE"

echo ""
echo "✅ Build complete!"
echo "   Version: $NEW_VERSION"
echo "   Zip: $ZIP_FILE"
echo "   Users will see update notification in extension popup"
