// Job Page Detector
// Runs on all supported job sites to detect job postings and inject UI
// Supports 20+ job sites including LinkedIn, Indeed, Greenhouse, Lever, Workday, etc.
// Cross-platform compatible: Windows, macOS, Linux

// Dynamic extractor loading based on current site
function getExtractor() {
  const url = window.location.href;
  const hostname = window.location.hostname;

  // LinkedIn - detect various job page patterns
  if (hostname.includes('linkedin.com')) {
    return {
      name: 'linkedin',
      isJobPage: () => {
        // Robust URL checks using regex
        if (/\/jobs\/view\/\d+/.test(url)) return true;
        if (url.includes('currentJobId=')) return true;
        if (/\/jobs\/collections\/recommended/.test(url) && url.includes('currentJobId=')) return true;

        // DOM checks - updated for 2024/2025 layout
        if (document.querySelector('.job-details-jobs-unified-top-card__job-title')) return true;
        if (document.querySelector('.jobs-unified-top-card__job-title')) return true;
        if (document.querySelector('.jobs-details__main-content')) return true;
        if (document.querySelector('[data-job-id]')) return true;

        // Fallback: Check for common job-specific elements
        if (document.querySelector('.jobs-description') && document.querySelector('.jobs-apply-button--top-card')) return true;

        return false;
      },
      extract: extractLinkedIn
    };
  }
  // Indeed - detect various job page patterns
  if (hostname.includes('indeed.com')) {
    return {
      name: 'indeed',
      isJobPage: () => {
        // URL patterns from research
        if (url.includes('/viewjob')) return true;
        if (url.includes('jk=')) return true;  // Direct job view
        if (url.includes('vjk=')) return true; // Job in search results
        // DOM checks
        if (document.querySelector('[data-testid="jobsearch-ViewJobLayout"]')) return true;
        if (document.querySelector('.jobsearch-JobInfoHeader-title')) return true;
        if (document.querySelector('.jobsearch-InlineCompanyRating')) return true;
        return false;
      },
      extract: extractIndeed
    };
  }
  // Glassdoor
  if (hostname.includes('glassdoor.com')) {
    return { name: 'glassdoor', isJobPage: () => url.includes('/job-listing/') || document.querySelector('[data-test="job-title"]'), extract: extractGlassdoor };
  }
  // ZipRecruiter
  if (hostname.includes('ziprecruiter.com')) {
    return { name: 'ziprecruiter', isJobPage: () => url.includes('/jobs/') || document.querySelector('.job_title'), extract: extractZipRecruiter };
  }
  // Greenhouse ATS
  if (hostname.includes('greenhouse.io') || hostname.includes('boards.greenhouse')) {
    return { name: 'greenhouse', isJobPage: () => url.includes('/jobs/') || document.querySelector('#app_form'), extract: extractGreenhouse };
  }
  // Lever ATS
  if (hostname.includes('lever.co') || hostname.includes('jobs.lever')) {
    return { name: 'lever', isJobPage: () => document.querySelector('.posting-headline'), extract: extractLever };
  }
  // Workday ATS
  if (hostname.includes('myworkdayjobs.com') || hostname.includes('workday.com')) {
    return { name: 'workday', isJobPage: () => document.querySelector('[data-automation-id="jobPostingHeader"]'), extract: extractWorkday };
  }
  // Monster
  if (hostname.includes('monster.com')) {
    return { name: 'monster', isJobPage: () => url.includes('/job-openings/') || document.querySelector('[data-testid="jobTitle"]'), extract: extractMonster };
  }
  // CareerBuilder
  if (hostname.includes('careerbuilder.com')) {
    return { name: 'careerbuilder', isJobPage: () => url.includes('/job/') || document.querySelector('.data-details-title'), extract: extractCareerBuilder };
  }
  // Wellfound (AngelList)
  if (hostname.includes('wellfound.com') || hostname.includes('angel.co')) {
    return { name: 'wellfound', isJobPage: () => url.includes('/jobs/') || url.includes('/role/'), extract: extractWellfound };
  }
  // Dice
  if (hostname.includes('dice.com')) {
    return { name: 'dice', isJobPage: () => url.includes('/job-detail/') || document.querySelector('[data-cy="jobTitle"]'), extract: extractDice };
  }
  // RemoteOK
  if (hostname.includes('remoteok.com') || hostname.includes('remoteok.io')) {
    return { name: 'remoteok', isJobPage: () => document.querySelector('.job, [itemtype*="JobPosting"]'), extract: extractRemoteOK };
  }
  // WeWorkRemotely
  if (hostname.includes('weworkremotely.com')) {
    return { name: 'weworkremotely', isJobPage: () => url.includes('/remote-jobs/'), extract: extractWeWorkRemotely };
  }
  // FlexJobs
  if (hostname.includes('flexjobs.com')) {
    return { name: 'flexjobs', isJobPage: () => url.includes('/jobs/'), extract: extractFlexJobs };
  }
  // iCIMS
  if (hostname.includes('icims.com') || document.querySelector('.iCIMS_JobHeader')) {
    return { name: 'icims', isJobPage: () => document.querySelector('.iCIMS_JobTitle'), extract: extractICIMS };
  }
  // BambooHR
  if (hostname.includes('bamboohr.com')) {
    return { name: 'bamboohr', isJobPage: () => document.querySelector('[data-testid="job-details"]'), extract: extractBambooHR };
  }
  // SmartRecruiters
  if (hostname.includes('smartrecruiters.com')) {
    return { name: 'smartrecruiters', isJobPage: () => document.querySelector('.job-title'), extract: extractSmartRecruiters };
  }
  // JazzHR
  if (hostname.includes('applytojob.com') || hostname.includes('jazz.co')) {
    return { name: 'jazzhr', isJobPage: () => document.querySelector('#job-title'), extract: extractJazzHR };
  }
  // BuiltIn
  if (hostname.includes('builtin.com')) {
    return { name: 'builtin', isJobPage: () => url.includes('/job/'), extract: extractBuiltIn };
  }
  // SimplyHired
  if (hostname.includes('simplyhired.com')) {
    return { name: 'simplyhired', isJobPage: () => url.includes('/job/') || document.querySelector('.viewjob-content'), extract: extractSimplyHired };
  }
  // The Muse
  if (hostname.includes('themuse.com')) {
    return { name: 'themuse', isJobPage: () => url.includes('/jobs/'), extract: extractTheMuse };
  }
  // Hired
  if (hostname.includes('hired.com')) {
    return { name: 'hired', isJobPage: () => url.includes('/jobs/'), extract: extractHired };
  }
  // Generic fallback for any other site
  return { name: 'generic', isJobPage: checkGenericJobPage, extract: extractGeneric };
}

// LinkedIn extraction (inline for content script) - Updated with 2024/2025 selectors
function extractLinkedIn() {
  // Verified selectors from browser research
  const titleSelectors = [
    'h1.top-card-layout__title',           // Public job page
    '.topcard__title',                      // Alternative layout
    '.job-details-jobs-unified-top-card__job-title',
    '.jobs-unified-top-card__job-title',
    'h1.t-24',
    '.jobs-details-top-card__job-title'
  ];
  const companySelectors = [
    'a.topcard__org-name-link',            // Public job page company link
    '.topcard__flavor--black-link',
    '.job-details-jobs-unified-top-card__company-name',
    '.jobs-unified-top-card__company-name',
    '.jobs-details-top-card__company-url'
  ];
  const locationSelectors = [
    '.topcard__flavor:not(.topcard__flavor--black-link)',  // Location text
    '.job-details-jobs-unified-top-card__bullet',
    '.jobs-unified-top-card__bullet',
    '.jobs-details-top-card__bullet'
  ];
  const descSelectors = [
    '.description__text',                   // Public job description
    '.show-more-less-html__markup',
    '.jobs-description-content__text',
    '.jobs-description__content',
    '#job-details'
  ];

  return {
    success: true,
    site: 'linkedin',
    title: findText(titleSelectors),
    company: findText(companySelectors),
    location: findText(locationSelectors),
    description: findText(descSelectors),
    hasEasyApply: !!document.querySelector('[data-control-name="jobdetails_topcard_inapply"], .jobs-apply-button--top-card, .jobs-apply-button'),
    jobUrl: window.location.href
  };
}

// Indeed extraction (inline)
function extractIndeed() {
  const titleSelectors = ['[data-testid="jobsearch-JobInfoHeader-title"]', '.jobsearch-JobInfoHeader-title', 'h1.icl-u-xs-mb--xs'];
  const companySelectors = ['[data-testid="inlineHeader-companyName"]', '.jobsearch-InlineCompanyRating a'];
  const locationSelectors = ['[data-testid="job-location"]', '[data-testid="jobsearch-JobInfoHeader-companyLocation"]'];
  const descSelectors = ['#jobDescriptionText', '.jobsearch-jobDescriptionText'];

  return {
    success: true,
    site: 'indeed',
    title: findText(titleSelectors),
    company: findText(companySelectors),
    location: findText(locationSelectors),
    description: findText(descSelectors),
    hasDirectApply: !!document.querySelector('#indeedApplyButton'),
    jobUrl: window.location.href
  };
}

// Glassdoor extraction
function extractGlassdoor() {
  return {
    success: true, site: 'glassdoor',
    title: findText(['[data-test="job-title"]', '.JobTitle', 'h1']),
    company: findText(['[data-test="employer-name"]', '.EmployerName']),
    location: findText(['[data-test="location"]', '.JobLocation']),
    description: findText(['.jobDescriptionContent', '.JobDescription']),
    hasDirectApply: !!document.querySelector('.gd-apply-button, .applyButton'),
    jobUrl: window.location.href
  };
}

// ZipRecruiter extraction
function extractZipRecruiter() {
  return {
    success: true, site: 'ziprecruiter',
    title: findText(['.job_title', 'h1.title', '.JobTitle']),
    company: findText(['.hiring_company', '.company_name', '.CompanyName']),
    location: findText(['.location', '.job_location']),
    description: findText(['.job_description', '#job_desc', '.Description']),
    hasDirectApply: !!document.querySelector('.apply_button, #quick_apply'),
    jobUrl: window.location.href
  };
}

// Greenhouse ATS extraction
function extractGreenhouse() {
  const company = window.location.pathname.match(/^\/([^\/]+)/)?.[1]?.replace(/-/g, ' ') || '';
  return {
    success: true, site: 'greenhouse',
    title: findText(['.app-title', 'h1.heading', '[class*="job-title"]']),
    company: company || findText(['.company-name']),
    location: findText(['.location', '[class*="location"]']),
    description: findText(['#content', '.content', '[class*="description"]']),
    hasDirectApply: !!document.querySelector('#application_form, #app_form'),
    jobUrl: window.location.href
  };
}

// Lever ATS extraction
function extractLever() {
  const company = window.location.pathname.match(/^\/([^\/]+)/)?.[1]?.replace(/-/g, ' ') || '';
  return {
    success: true, site: 'lever',
    title: findText(['.posting-headline h2', '[data-qa="posting-name"]']),
    company: company || findText(['.posting-categories .location']),
    location: findText(['.posting-categories .location', '[data-qa="posting-location"]']),
    description: findText(['.posting-description', '[data-qa="posting-description"]']),
    hasDirectApply: !!document.querySelector('.postings-btn-wrapper a, [data-qa="btn-apply"]'),
    jobUrl: window.location.href
  };
}

// Workday ATS extraction
function extractWorkday() {
  const company = window.location.hostname.match(/^([^.]+)\.wd/)?.[1]?.replace(/-/g, ' ') || '';
  return {
    success: true, site: 'workday',
    title: findText(['[data-automation-id="jobPostingHeader"]', '.css-1q2dra3']),
    company: company,
    location: findText(['[data-automation-id="locations"]', '[data-automation-id="jobPostingLocation"]']),
    description: findText(['[data-automation-id="jobPostingDescription"]', '.job-description']),
    hasDirectApply: !!document.querySelector('[data-automation-id="applyButton"]'),
    jobUrl: window.location.href
  };
}

// Monster extraction
function extractMonster() {
  return {
    success: true, site: 'monster',
    title: findText(['[data-testid="jobTitle"]', '.JobViewTitle', 'h1']),
    company: findText(['[data-testid="company"]', '.company-name']),
    location: findText(['[data-testid="jobDetailLocation"]', '.location']),
    description: findText(['[data-testid="jobDescription"]', '.job-description']),
    hasDirectApply: !!document.querySelector('[data-testid="applyButton"]'),
    jobUrl: window.location.href
  };
}

// CareerBuilder extraction
function extractCareerBuilder() {
  return {
    success: true, site: 'careerbuilder',
    title: findText(['.data-details-title', 'h1.job-title']),
    company: findText(['.data-details-company', '.company-name']),
    location: findText(['.data-details-location', '.job-location']),
    description: findText(['.data-details-description', '.job-description']),
    hasDirectApply: !!document.querySelector('.data-details-apply'),
    jobUrl: window.location.href
  };
}

// Wellfound (AngelList) extraction
function extractWellfound() {
  return {
    success: true, site: 'wellfound',
    title: findText(['[data-test="JobTitle"]', 'h1[class*="title"]']),
    company: findText(['[data-test="CompanyName"]', 'a[class*="company"]']),
    location: findText(['[data-test="Location"]', '[class*="location"]']),
    description: findText(['[data-test="JobDescription"]', '.job-description']),
    hasDirectApply: !!document.querySelector('button[class*="apply"]'),
    jobUrl: window.location.href
  };
}

// Dice extraction
function extractDice() {
  return {
    success: true, site: 'dice',
    title: findText(['[data-cy="jobTitle"]', 'h1.jobTitle']),
    company: findText(['[data-cy="companyName"]', '.employer-name']),
    location: findText(['[data-cy="location"]', '.location']),
    description: findText(['[data-cy="jobDescription"]', '.job-description']),
    hasDirectApply: !!document.querySelector('[data-cy="applyButton"]'),
    jobUrl: window.location.href
  };
}

// RemoteOK extraction
function extractRemoteOK() {
  return {
    success: true, site: 'remoteok',
    title: findText(['[itemprop="title"]', 'h2', '.company a:last-child']),
    company: findText(['[itemprop="hiringOrganization"]', 'h3.company']),
    location: 'Remote',
    description: findText(['[itemprop="description"]', '.description', '.markdown']),
    hasDirectApply: !!document.querySelector('.apply, a[href*="apply"]'),
    jobUrl: window.location.href
  };
}

// WeWorkRemotely extraction
function extractWeWorkRemotely() {
  return {
    success: true, site: 'weworkremotely',
    title: findText(['h1', '.listing-header-container h1']),
    company: findText(['.company', '.listing-header-container .company']),
    location: 'Remote',
    description: findText(['.listing-container', '.job-description']),
    hasDirectApply: !!document.querySelector('.apply-button'),
    jobUrl: window.location.href
  };
}

// FlexJobs extraction
function extractFlexJobs() {
  return {
    success: true, site: 'flexjobs',
    title: findText(['h1.job-title', '.job-heading h1']),
    company: findText(['.job-company', '.company-name']),
    location: findText(['.job-location', '.location']),
    description: findText(['.job-description', '#job-description']),
    hasDirectApply: !!document.querySelector('.apply-btn'),
    jobUrl: window.location.href
  };
}

// iCIMS extraction
function extractICIMS() {
  return {
    success: true, site: 'icims',
    title: findText(['.iCIMS_JobTitle', 'h1']),
    company: findText(['.iCIMS_Company']),
    location: findText(['.iCIMS_JobLocation']),
    description: findText(['.iCIMS_JobDescription', '.iCIMS_ExpandableText']),
    hasDirectApply: !!document.querySelector('.iCIMS_ApplyButton'),
    jobUrl: window.location.href
  };
}

// BambooHR extraction
function extractBambooHR() {
  return {
    success: true, site: 'bamboohr',
    title: findText(['[data-testid="job-title"]', 'h2.JobTitle', 'h1']),
    company: findText(['[data-testid="company-name"]']),
    location: findText(['[data-testid="job-location"]', '.location']),
    description: findText(['[data-testid="job-description"]', '.JobDesc']),
    hasDirectApply: !!document.querySelector('[data-testid="apply-button"]'),
    jobUrl: window.location.href
  };
}

// SmartRecruiters extraction
function extractSmartRecruiters() {
  return {
    success: true, site: 'smartrecruiters',
    title: findText(['h1.job-title', '[data-automation="job-title"]']),
    company: findText(['.company-name', '[data-automation="company-name"]']),
    location: findText(['.job-location', '[data-automation="job-location"]']),
    description: findText(['.job-description', '[data-automation="job-description"]']),
    hasDirectApply: !!document.querySelector('.apply-btn'),
    jobUrl: window.location.href
  };
}

// JazzHR extraction
function extractJazzHR() {
  return {
    success: true, site: 'jazzhr',
    title: findText(['#job-title', 'h1.job-title']),
    company: findText(['.company-name']),
    location: findText(['#job-location', '.location']),
    description: findText(['#job-description', '.description']),
    hasDirectApply: !!document.querySelector('#apply-button'),
    jobUrl: window.location.href
  };
}

// BuiltIn extraction
function extractBuiltIn() {
  return {
    success: true, site: 'builtin',
    title: findText(['h1.job-title', '[data-id="job-title"]']),
    company: findText(['.company-name', '[data-id="company-name"]']),
    location: findText(['.location', '[data-id="location"]']),
    description: findText(['.job-description', '[data-id="job-description"]']),
    hasDirectApply: !!document.querySelector('.apply-button'),
    jobUrl: window.location.href
  };
}

// SimplyHired extraction
function extractSimplyHired() {
  return {
    success: true, site: 'simplyhired',
    title: findText(['h1.viewjob-jobTitle', 'h2[data-testid="viewJobTitle"]']),
    company: findText(['.viewjob-company', '[data-testid="viewJobCompany"]']),
    location: findText(['.viewjob-labelWithIcon', '[data-testid="viewJobLocation"]']),
    description: findText(['.viewjob-description', '[data-testid="viewJobDescription"]']),
    hasDirectApply: !!document.querySelector('.viewjob-apply'),
    jobUrl: window.location.href
  };
}

// TheMuse extraction
function extractTheMuse() {
  return {
    success: true, site: 'themuse',
    title: findText(['h1.job-board-job-title']),
    company: findText(['.company-name', '.job-board-company']),
    location: findText(['.job-location', '.location']),
    description: findText(['.job-description', '.job-board-description']),
    hasDirectApply: !!document.querySelector('.apply-button'),
    jobUrl: window.location.href
  };
}

// Hired extraction
function extractHired() {
  return {
    success: true, site: 'hired',
    title: findText(['h1[class*="title"]', '.job-title']),
    company: findText(['[class*="company"]', '.company-name']),
    location: findText(['[class*="location"]']),
    description: findText(['[class*="description"]', '.job-description']),
    hasDirectApply: true,
    jobUrl: window.location.href
  };
}

// Generic extraction
function extractGeneric() {
  const title = findText(['h1', '[class*="job-title"]', '[class*="jobTitle"]']) || document.title;
  const company = findText(['[class*="company"]', '[class*="employer"]']) || '';
  const location = findText(['[class*="location"]']) || '';

  let description = '';
  const descEl = document.querySelector('[class*="description"], article, main');
  if (descEl) description = descEl.textContent.trim().slice(0, 5000);

  return {
    success: true,
    site: 'generic',
    title,
    company,
    location,
    description,
    jobUrl: window.location.href
  };
}

// Universal job form detection - works on any site
function checkGenericJobPage() {
  let score = 0;
  const url = window.location.href.toLowerCase();
  const title = document.title.toLowerCase();
  const body = document.body;
  if (!body) return false;

  const text = body.textContent.toLowerCase();

  // URL signals (low weight - as user requested, not just URL-based)
  const urlKeywords = ['job', 'career', 'apply', 'position', 'hiring', 'recruit', 'talent', 'opportunity'];
  if (urlKeywords.some(k => url.includes(k))) score += 1;

  // Title signals
  const titleKeywords = ['job', 'career', 'apply', 'position', 'hiring', 'opening', 'opportunity', 'employment'];
  if (titleKeywords.some(k => title.includes(k))) score += 2;

  // === DOM-BASED DETECTION (high weight) ===

  // Check for job application forms by looking for common form elements
  const forms = document.querySelectorAll('form');
  for (const form of forms) {
    const formText = form.textContent.toLowerCase();
    const formHTML = form.innerHTML.toLowerCase();

    // Check for job-related form fields
    const hasNameField = form.querySelector('input[name*="name"], input[id*="name"], input[placeholder*="name" i]');
    const hasEmailField = form.querySelector('input[type="email"], input[name*="email"], input[id*="email"]');
    const hasPhoneField = form.querySelector('input[type="tel"], input[name*="phone"], input[id*="phone"]');
    const hasResumeUpload = form.querySelector('input[type="file"], input[name*="resume"], input[name*="cv"], [id*="resume"], [id*="cv"]');

    // Strong signal: resume/CV upload
    if (hasResumeUpload) score += 5;

    // Moderate signal: typical application form fields
    if (hasNameField && hasEmailField) score += 3;
    if (hasPhoneField) score += 1;

    // Check form action/text for job-related terms
    const formAction = (form.action || '').toLowerCase();
    if (['apply', 'submit', 'application', 'candidate'].some(k => formAction.includes(k))) score += 2;
    if (['apply', 'submit application', 'send application'].some(k => formText.includes(k))) score += 2;
  }

  // Check for standalone file upload (outside forms)
  const resumeInputs = document.querySelectorAll('input[type="file"]');
  for (const input of resumeInputs) {
    const nearbyText = (input.closest('div, section, label')?.textContent || '').toLowerCase();
    if (['resume', 'cv', 'curriculum', 'upload'].some(k => nearbyText.includes(k))) {
      score += 4;
    }
  }

  // Check for Apply buttons
  const applyButtons = document.querySelectorAll('button, a, input[type="submit"]');
  for (const btn of applyButtons) {
    const btnText = btn.textContent.toLowerCase().trim();
    const btnValue = (btn.value || '').toLowerCase();
    const combined = btnText + ' ' + btnValue;

    if (['apply now', 'apply for this job', 'apply for job', 'submit application', 'apply'].some(k => combined.includes(k))) {
      score += 3;
      break;
    }
  }

  // Check for job listing content
  const contentKeywords = [
    { term: 'job description', weight: 2 },
    { term: 'responsibilities', weight: 2 },
    { term: 'qualifications', weight: 2 },
    { term: 'requirements', weight: 2 },
    { term: 'about the role', weight: 2 },
    { term: 'what you\'ll do', weight: 2 },
    { term: 'who you are', weight: 1 },
    { term: 'experience required', weight: 2 },
    { term: 'salary', weight: 1 },
    { term: 'benefits', weight: 1 },
    { term: 'apply now', weight: 2 },
    { term: 'submit your resume', weight: 3 },
    { term: 'cover letter', weight: 2 },
    { term: 'full-time', weight: 1 },
    { term: 'part-time', weight: 1 },
    { term: 'remote work', weight: 1 },
    { term: 'hybrid', weight: 1 },
  ];

  for (const kw of contentKeywords) {
    if (text.includes(kw.term)) score += kw.weight;
  }

  // Check for common ATS platform elements
  const atsSelectors = [
    '#app_form', // Greenhouse
    '.posting-headline', // Lever  
    '[data-automation-id="jobPostingHeader"]', // Workday
    '#st-job-title', // SmartRecruiters
    '.icims-container', // iCIMS
    '.application-form',
    '#job-application-form',
    '.job-apply-form',
  ];

  if (atsSelectors.some(sel => document.querySelector(sel))) {
    score += 5;
  }

  console.log(`[Eric's AutoApply] Generic detection score: ${score}`);

  // Threshold: 5 points for moderate confidence, 8+ for high confidence
  return score >= 5;
}

// Helper to find text from multiple selectors
function findText(selectors) {
  for (const sel of selectors) {
    const el = document.querySelector(sel);
    if (el) {
      const text = el.textContent.trim();
      if (text) return text;
    }
  }
  return '';
}

// State
let fabInjected = false;
let currentJobData = null;

// Main detection function - ALWAYS scans page, auto-fills basic fields, detects resume upload
async function detectJob() {
  const extractor = getExtractor();
  const url = window.location.href;

  console.log("[Eric's AutoApply] Running detection...");
  console.log("[Eric's AutoApply] URL:", url);

  // ALWAYS analyze page for fillable fields first
  let analysis = null;
  let fieldCount = 0;
  let hasResumeUpload = false;

  if (window.PageAnalyzer) {
    analysis = window.PageAnalyzer.analyze();
    fieldCount = analysis.totalFillableFields || 0;
    hasResumeUpload = analysis.hasResumeUpload;
    console.log(`[Eric's AutoApply] Found ${fieldCount} fillable fields, resume upload: ${hasResumeUpload}`);
  }

  // Try to extract job data (for AI generation)
  currentJobData = extractor.extract();
  if (!currentJobData.title) {
    currentJobData = extractGenericForced();
  }
  console.log("[Eric's AutoApply] Extracted data:", currentJobData?.title || 'No title');

  // AUTO-FILL basic fields immediately (if we have profile data)
  if (fieldCount > 0 && window.SmartAutofill) {
    try {
      const initialized = await window.SmartAutofill.init();
      if (initialized) {
        const fillResult = await window.SmartAutofill.fillPage(analysis);
        console.log(`[Eric's AutoApply] Auto-filled ${fillResult.filled} fields`);

        // Show notification if we filled anything
        if (fillResult.filled > 0) {
          showNotification(`⚡ Auto-filled ${fillResult.filled} fields`);
        }
      }
    } catch (e) {
      console.log("[Eric's AutoApply] Auto-fill skipped:", e.message);
    }
  }

  // Show FAB with remaining options
  if (fieldCount > 0 || (currentJobData?.success && currentJobData?.title)) {
    console.log(`[Eric's AutoApply] Showing FAB: ${fieldCount} fields`);
    injectFAB(fieldCount);
  }

  // SMART RESUME DETECTION: If we found a resume upload field, show the generation prompt
  if (hasResumeUpload && analysis.resumeUploadElement) {
    console.log("[Eric's AutoApply] Resume upload detected - showing smart prompt");
    showSmartResumePrompt(analysis.resumeUploadElement);
  }
}

// Smart resume generation prompt - appears when resume/CV upload is detected
function showSmartResumePrompt(fileInput) {
  // Don't show if already shown
  if (document.getElementById('ea-smart-resume-prompt')) return;

  const prompt = document.createElement('div');
  prompt.id = 'ea-smart-resume-prompt';
  prompt.innerHTML = `
    <div class="ea-smart-prompt-overlay">
      <div class="ea-smart-prompt-content">
        <div class="ea-smart-prompt-header">
          <span class="ea-smart-icon">📄</span>
          <div>
            <h3>Resume Upload Detected</h3>
            <p>Generate a tailored resume for this application?</p>
          </div>
          <button class="ea-smart-close">×</button>
        </div>
        <div class="ea-smart-prompt-body">
          <div class="ea-job-context">
            <strong>${currentJobData?.title || 'Position'}</strong>
            <span>${currentJobData?.company || ''}</span>
          </div>
          <div class="ea-smart-input-section">
            <label>💬 Notes for this specific job (optional):</label>
            <textarea id="ea-job-specific-notes" rows="3" placeholder="E.g., They're looking for someone with Python experience - emphasize my Django projects. Mention I'm excited about their AI focus."></textarea>
          </div>
          <div class="ea-smart-actions">
            <button class="ea-smart-generate">✨ Generate & Upload Resume</button>
            <button class="ea-smart-skip">Skip</button>
          </div>
        </div>
      </div>
    </div>
  `;

  // Add styles
  if (!document.getElementById('ea-smart-prompt-styles')) {
    const style = document.createElement('style');
    style.id = 'ea-smart-prompt-styles';
    style.textContent = `
      .ea-smart-prompt-overlay {
        position: fixed;
        bottom: 100px;
        right: 20px;
        z-index: 2147483647;
        animation: ea-slide-up 0.3s ease;
      }
      @keyframes ea-slide-up {
        from { opacity: 0; transform: translateY(20px); }
        to { opacity: 1; transform: translateY(0); }
      }
      .ea-smart-prompt-content {
        background: linear-gradient(135deg, #1e1e2e 0%, #0a0a14 100%);
        border: 1px solid #374151;
        border-radius: 16px;
        width: 360px;
        box-shadow: 0 8px 32px rgba(0,0,0,0.5);
        overflow: hidden;
      }
      .ea-smart-prompt-header {
        display: flex;
        align-items: flex-start;
        gap: 12px;
        padding: 16px;
        background: rgba(34, 197, 94, 0.1);
        border-bottom: 1px solid #374151;
      }
      .ea-smart-icon {
        font-size: 28px;
      }
      .ea-smart-prompt-header h3 {
        margin: 0;
        color: #22c55e;
        font-size: 15px;
      }
      .ea-smart-prompt-header p {
        margin: 4px 0 0 0;
        color: #9ca3af;
        font-size: 13px;
      }
      .ea-smart-close {
        margin-left: auto;
        background: none;
        border: none;
        color: #6b7280;
        font-size: 20px;
        cursor: pointer;
        padding: 0;
        line-height: 1;
      }
      .ea-smart-close:hover { color: #fff; }
      .ea-smart-prompt-body {
        padding: 16px;
      }
      .ea-job-context {
        background: rgba(255,255,255,0.05);
        padding: 10px 12px;
        border-radius: 8px;
        margin-bottom: 12px;
      }
      .ea-job-context strong {
        display: block;
        color: #fff;
        font-size: 14px;
      }
      .ea-job-context span {
        color: #9ca3af;
        font-size: 12px;
      }
      .ea-smart-input-section {
        margin-bottom: 16px;
      }
      .ea-smart-input-section label {
        display: block;
        color: #9ca3af;
        font-size: 12px;
        margin-bottom: 8px;
      }
      .ea-smart-input-section textarea {
        width: 100%;
        padding: 10px;
        background: #0a0a14;
        border: 1px solid #374151;
        border-radius: 8px;
        color: #fff;
        font-size: 13px;
        resize: vertical;
        font-family: inherit;
      }
      .ea-smart-input-section textarea:focus {
        outline: none;
        border-color: #22c55e;
      }
      .ea-smart-actions {
        display: flex;
        gap: 8px;
      }
      .ea-smart-generate {
        flex: 1;
        padding: 12px 16px;
        background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
        border: none;
        border-radius: 8px;
        color: #000;
        font-size: 13px;
        font-weight: 600;
        cursor: pointer;
        transition: transform 0.2s;
      }
      .ea-smart-generate:hover { transform: scale(1.02); }
      .ea-smart-generate:disabled {
        opacity: 0.6;
        cursor: not-allowed;
        transform: none;
      }
      .ea-smart-skip {
        padding: 12px 16px;
        background: transparent;
        border: 1px solid #374151;
        border-radius: 8px;
        color: #9ca3af;
        font-size: 13px;
        cursor: pointer;
      }
      .ea-smart-skip:hover { border-color: #6b7280; color: #fff; }
    `;
    document.head.appendChild(style);
  }

  // Event handlers
  prompt.querySelector('.ea-smart-close').addEventListener('click', () => prompt.remove());
  prompt.querySelector('.ea-smart-skip').addEventListener('click', () => prompt.remove());

  prompt.querySelector('.ea-smart-generate').addEventListener('click', async (e) => {
    const btn = e.target;
    const jobNotes = document.getElementById('ea-job-specific-notes').value;

    btn.textContent = '⏳ Generating...';
    btn.disabled = true;

    try {
      // Check auth first
      const authResponse = await chrome.runtime.sendMessage({ action: 'checkAuth' });
      if (!authResponse.isAuthenticated) {
        prompt.remove();
        showPanel('login');
        return;
      }

      // Get user's settings and master resume
      const userData = await chrome.storage.local.get([
        'masterResume',
        'additionalInfo',  // Global notes from settings
        'userProfile'
      ]);
      const settings = await chrome.runtime.sendMessage({ action: 'getSettings' });

      const masterResume = userData.masterResume || '';
      const globalNotes = userData.additionalInfo || userData.userProfile?.generationNotes || '';
      const preferredFormat = settings.settings?.preferredFormat || 'pdf';
      const preferredStyle = settings.settings?.preferredStyle || 'professional-modern';

      // Generate resume with layered comments
      const response = await chrome.runtime.sendMessage({
        action: 'generate',
        data: {
          title: currentJobData?.title || 'Position',
          company: currentJobData?.company || '',
          description: currentJobData?.description || '',
          location: currentJobData?.location || '',
          jobUrl: currentJobData?.jobUrl || window.location.href,
          masterResume: masterResume,
          additionalInfo: [globalNotes, jobNotes].filter(n => n.trim()).join('\n\n---\n\n')
        }
      });

      if (!response.success) {
        throw new Error(response.error || 'Generation failed');
      }

      // Download the resume in user's preferred format
      const downloadResponse = await chrome.runtime.sendMessage({
        action: 'download',
        data: {
          type: 'resume',
          content: response.resume,
          format: preferredFormat,
          style: preferredStyle
        }
      });

      if (downloadResponse.success) {
        // Create and trigger download
        const a = document.createElement('a');
        a.href = downloadResponse.data;
        a.download = downloadResponse.filename;
        a.click();

        showNotification(`✅ Resume generated! Upload the downloaded ${preferredFormat.toUpperCase()} file.`);

        // Highlight the file input
        if (fileInput) {
          fileInput.style.boxShadow = '0 0 0 3px #22c55e';
          fileInput.style.transition = 'box-shadow 0.3s';
          setTimeout(() => {
            fileInput.style.boxShadow = '';
          }, 3000);
        }
      }

      prompt.remove();

    } catch (error) {
      console.error('[Eric\'s AutoApply] Resume generation error:', error);
      showNotification(`❌ Error: ${error.message}`);
      btn.textContent = '✨ Generate & Upload Resume';
      btn.disabled = false;
    }
  });

  document.body.appendChild(prompt);
}


// Inject floating action button with field count
function injectFAB(fieldCount = 0) {
  if (fabInjected) {
    // Update existing FAB count
    const countBadge = document.querySelector('.ea-fab-count');
    if (countBadge && fieldCount > 0) {
      countBadge.textContent = fieldCount;
      countBadge.style.display = 'flex';
    }
    return;
  }

  // Create FAB container
  const fab = document.createElement('div');
  fab.id = 'ea-fab';
  fab.innerHTML = `
    <div class="ea-fab-button" title="${fieldCount > 0 ? `${fieldCount} fields ready to fill` : 'Generate tailored application'}">
      <svg viewBox="0 0 24 24" width="24" height="24" fill="currentColor">
        <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/>
      </svg>
      ${fieldCount > 0 ? `<span class="ea-fab-count">${fieldCount}</span>` : ''}
    </div>
    <div class="ea-fab-pulse"></div>
    <div class="ea-fab-menu" style="display:none;">
      <button class="ea-fab-menu-item ea-fill-now" title="Fill all fields with your profile data">
        ⚡ Fill Page
      </button>
      <button class="ea-fab-menu-item ea-generate" title="Generate AI-tailored content">
        ✨ AI Generate
      </button>
    </div>
  `;

  // Add styles for count badge and menu
  if (!document.getElementById('ea-fab-extra-styles')) {
    const style = document.createElement('style');
    style.id = 'ea-fab-extra-styles';
    style.textContent = `
      .ea-fab-count {
        position: absolute;
        top: -4px;
        right: -4px;
        background: #22c55e;
        color: #fff;
        font-size: 11px;
        font-weight: bold;
        min-width: 18px;
        height: 18px;
        border-radius: 9px;
        display: flex;
        align-items: center;
        justify-content: center;
        padding: 0 4px;
        box-shadow: 0 2px 4px rgba(0,0,0,0.3);
      }
      .ea-fab-menu {
        position: absolute;
        bottom: 60px;
        right: 0;
        background: linear-gradient(135deg, #1e1e2e 0%, #0a0a14 100%);
        border: 1px solid #374151;
        border-radius: 12px;
        padding: 8px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.5);
        min-width: 140px;
      }
      .ea-fab-menu-item {
        display: block;
        width: 100%;
        padding: 10px 14px;
        margin: 4px 0;
        background: transparent;
        border: 1px solid transparent;
        border-radius: 8px;
        color: #fff;
        font-size: 13px;
        text-align: left;
        cursor: pointer;
        transition: all 0.2s;
      }
      .ea-fab-menu-item:hover {
        background: rgba(34, 197, 94, 0.2);
        border-color: #22c55e;
      }
      .ea-fab-menu-item.ea-fill-now {
        background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
        font-weight: 600;
      }
      .ea-fab-menu-item.ea-fill-now:hover {
        transform: scale(1.02);
      }
    `;
    document.head.appendChild(style);
  }

  // Toggle menu on FAB click
  fab.querySelector('.ea-fab-button').addEventListener('click', (e) => {
    const menu = fab.querySelector('.ea-fab-menu');
    const isVisible = menu.style.display !== 'none';
    menu.style.display = isVisible ? 'none' : 'block';
  });

  // Fill Now button - quick fill without AI
  fab.querySelector('.ea-fill-now').addEventListener('click', async (e) => {
    e.stopPropagation();
    fab.querySelector('.ea-fab-menu').style.display = 'none';

    const btn = e.target;
    btn.textContent = '⏳ Filling...';
    btn.disabled = true;

    try {
      // Initialize SmartAutofill and fill the page
      if (window.SmartAutofill && window.PageAnalyzer) {
        await window.SmartAutofill.init();
        const analysis = window.PageAnalyzer.analyze();
        const result = await window.SmartAutofill.fillPage(analysis);

        if (result.success && result.filled > 0) {
          showNotification(`✅ Filled ${result.filled} fields!`);
        } else if (result.filled === 0) {
          showNotification('⚠️ No fields to fill. Set up your profile first.');
        } else {
          showNotification(`⚠️ ${result.error || 'Some fields may need attention'}`);
        }
      } else {
        showNotification('❌ Autofill modules not loaded');
      }
    } catch (error) {
      showNotification(`❌ Error: ${error.message}`);
    } finally {
      btn.textContent = '⚡ Fill Page';
      btn.disabled = false;
    }
  });

  // AI Generate button - opens the full panel
  fab.querySelector('.ea-generate').addEventListener('click', (e) => {
    e.stopPropagation();
    fab.querySelector('.ea-fab-menu').style.display = 'none';
    openQuickApply();
  });

  // Close menu when clicking outside
  document.addEventListener('click', (e) => {
    if (!fab.contains(e.target)) {
      fab.querySelector('.ea-fab-menu').style.display = 'none';
    }
  });

  document.body.appendChild(fab);
  fabInjected = true;
}


// Remove FAB
function removeFAB() {
  const fab = document.getElementById('ea-fab');
  if (fab) {
    fab.remove();
    fabInjected = false;
  }
}

// Open quick apply panel
async function openQuickApply() {
  // Check if already authenticated
  const authResponse = await chrome.runtime.sendMessage({ action: 'checkAuth' });

  if (!authResponse.isAuthenticated) {
    // Show login prompt
    showPanel('login');
    return;
  }

  // Show quick apply panel with job data
  showPanel('generate', currentJobData);
}

// Show panel overlay
function showPanel(type, data = {}) {
  // Remove existing panel
  const existing = document.getElementById('ea-panel');
  if (existing) existing.remove();

  // Create panel
  const panel = document.createElement('div');
  panel.id = 'ea-panel';

  if (type === 'login') {
    panel.innerHTML = createLoginPanel();
  } else if (type === 'generate') {
    panel.innerHTML = createGeneratePanel(data);
  } else if (type === 'result') {
    panel.innerHTML = createResultPanel(data);
  }

  document.body.appendChild(panel);

  // Add event listeners
  setupPanelListeners(type, data);
}

function createLoginPanel() {
  return `
    <div class="ea-panel-overlay">
      <div class="ea-panel-content">
        <div class="ea-panel-header">
          <h2>⚡ exploit.team AutoJob</h2>
          <button class="ea-close" onclick="document.getElementById('ea-panel').remove()">×</button>
        </div>
        <div class="ea-panel-body">
          <p>Login to your exploit.team account to generate tailored applications.</p>
          <form id="ea-login-form">
            <input type="text" id="ea-username" placeholder="Username" required>
            <input type="password" id="ea-password" placeholder="Password" required>
            <button type="submit" class="ea-btn-primary">Login</button>
          </form>
          <p class="ea-login-error" style="display:none; color:#ef4444;"></p>
          <p class="ea-signup-link">
            Don't have an account? <a href="https://job.exploit.team/register" target="_blank">Sign up</a>
          </p>
        </div>
      </div>
    </div>
  `;
}

function createGeneratePanel(data) {
  return `
    <div class="ea-panel-overlay">
      <div class="ea-panel-content ea-panel-wide">
        <div class="ea-panel-header">
          <h2>⚡ exploit.team AutoJob</h2>
          <button class="ea-close" onclick="document.getElementById('ea-panel').remove()">×</button>
        </div>
        <div class="ea-panel-body">
          <div class="ea-job-info">
            <h3>${data.title || 'Unknown Position'}</h3>
            <p class="ea-company">${data.company || 'Unknown Company'} • ${data.location || ''}</p>
          </div>
          <div class="ea-generate-section">
            <div class="ea-comments-section" style="margin-bottom:16px;">
              <label style="display:block;margin-bottom:8px;color:#9ca3af;font-size:13px;">💬 Additional notes for this job (optional):</label>
              <textarea id="ea-user-comments" rows="3" style="width:100%;padding:12px;background:#1a1a2e;border:1px solid #374151;border-radius:8px;color:#fff;font-size:14px;resize:vertical;" placeholder="Add specific points you want to emphasize, salary expectations, special skills to highlight, etc."></textarea>
            </div>
            <p>Generate tailored application materials:</p>
            <div class="ea-generate-options">
              <label><input type="checkbox" id="gen-resume" checked> Resume</label>
              <label><input type="checkbox" id="gen-cover" checked> Cover Letter</label>
              <label><input type="checkbox" id="gen-email" checked> Email Draft</label>
            </div>
            <button id="ea-generate-btn" class="ea-btn-primary ea-btn-large">
              ✨ Generate Application
            </button>
            <div id="ea-loading" style="display:none;">
              <div class="ea-progress-container" style="margin-bottom:16px;">
                <div class="ea-progress-bar" style="width:100%;height:8px;background:#1a1a2e;border-radius:4px;overflow:hidden;">
                  <div id="ea-progress-fill" style="width:0%;height:100%;background:linear-gradient(90deg,#22c55e,#16a34a);transition:width 0.3s ease;"></div>
                </div>
                <div id="ea-progress-steps" style="display:flex;justify-content:space-between;margin-top:8px;font-size:11px;color:#6b7280;">
                  <span id="step-1" class="ea-step">📄 Analyzing job</span>
                  <span id="step-2" class="ea-step">📝 Generating resume</span>
                  <span id="step-3" class="ea-step">✉️ Creating cover letter</span>
                  <span id="step-4" class="ea-step">✅ Finalizing</span>
                </div>
              </div>
              <p id="ea-progress-text" style="text-align:center;color:#9ca3af;">Analyzing job details...</p>
            </div>
          </div>
        </div>
      </div>
    </div>
  `;
}

function createResultPanel(data) {
  const defaultStyle = data.preferredStyle || 'professional-modern';
  const defaultFormat = data.preferredFormat || 'pdf';
  return `
    <div class="ea-panel-overlay">
      <div class="ea-panel-content ea-panel-wide">
        <div class="ea-panel-header">
          <h2>✅ Application Ready</h2>
          <button class="ea-close" onclick="document.getElementById('ea-panel').remove()">×</button>
        </div>
        <div class="ea-panel-body">
          <div class="ea-result-tabs">
            <button class="ea-tab active" data-tab="resume">Resume</button>
            <button class="ea-tab" data-tab="cover">Cover Letter</button>
            <button class="ea-tab" data-tab="email">Email</button>
            <button class="ea-tab" data-tab="askai">💬 Ask AI</button>
          </div>
          <div class="ea-tab-content" id="tab-resume">
            <pre>${data.resume || 'Not generated'}</pre>
          </div>
          <div class="ea-tab-content" id="tab-cover" style="display:none;">
            <pre>${data.coverLetter || 'Not generated'}</pre>
          </div>
          <div class="ea-tab-content" id="tab-email" style="display:none;">
            <pre>${data.emailDraft || 'Not generated'}</pre>
          </div>
          <div class="ea-tab-content" id="tab-askai" style="display:none;">
            <div class="ea-askai-section">
              <p style="margin-bottom:12px;color:#9ca3af;">Need help with a custom question? Paste it here and AI will generate an answer.</p>
              <textarea id="ea-question-input" placeholder="Paste the question here, e.g.:\n• Why do you want to work at [Company]?\n• What are your salary expectations?\n• Describe a time you showed leadership..." rows="3" style="width:100%;padding:12px;background:#1a1a2e;border:1px solid #374151;border-radius:8px;color:#fff;font-size:14px;resize:vertical;"></textarea>
              <button id="ea-generate-answer" class="ea-btn-primary" style="margin-top:12px;">✨ Generate Answer</button>
              <div id="ea-answer-loading" style="display:none;margin-top:12px;color:#9ca3af;">
                <span class="ea-spinner-small"></span> Generating...
              </div>
              <div id="ea-answer-result" style="display:none;margin-top:12px;">
                <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:8px;">
                  <strong style="color:#22c55e;">Generated Answer:</strong>
                  <button id="ea-copy-answer" class="ea-btn" style="padding:4px 12px;font-size:12px;">📋 Copy</button>
                </div>
                <pre id="ea-answer-text" style="background:#0a0a14;padding:12px;border-radius:8px;white-space:pre-wrap;max-height:200px;overflow-y:auto;"></pre>
              </div>
            </div>
          </div>
          <div class="ea-result-actions">
            <div class="ea-style-selector" style="display:flex;align-items:center;gap:16px;margin-bottom:12px;flex-wrap:wrap;">
              <div style="display:flex;align-items:center;gap:8px;">
                <label style="color:#9ca3af;font-size:13px;">Style:</label>
                <select id="ea-doc-style" style="background:#1a1a2e;color:#fff;border:1px solid #374151;border-radius:6px;padding:6px 12px;font-size:13px;">
                  <option value="professional-modern" ${defaultStyle === 'professional-modern' ? 'selected' : ''}>Professional Modern</option>
                  <option value="classic-traditional" ${defaultStyle === 'classic-traditional' ? 'selected' : ''}>Classic Traditional</option>
                  <option value="creative-bold" ${defaultStyle === 'creative-bold' ? 'selected' : ''}>Creative Bold</option>
                  <option value="simple-clean" ${defaultStyle === 'simple-clean' ? 'selected' : ''}>Simple Clean</option>
                  <option value="executive" ${defaultStyle === 'executive' ? 'selected' : ''}>Executive</option>
                </select>
              </div>
              <div style="display:flex;align-items:center;gap:8px;">
                <label style="color:#9ca3af;font-size:13px;">Format:</label>
                <select id="ea-doc-format" style="background:#1a1a2e;color:#fff;border:1px solid #374151;border-radius:6px;padding:6px 12px;font-size:13px;">
                  <option value="pdf" ${defaultFormat === 'pdf' ? 'selected' : ''}>📄 PDF</option>
                  <option value="docx" ${defaultFormat === 'docx' ? 'selected' : ''}>📝 DOCX</option>
                </select>
              </div>
            </div>
            <div style="display:flex;gap:8px;flex-wrap:wrap;">
              <button class="ea-btn" id="ea-copy">📋 Copy</button>
              <button class="ea-btn-primary" id="ea-download">⬇️ Download</button>
              ${data.hasEasyApply ? '<button class="ea-btn-primary" id="ea-autofill">🚀 Auto-Fill</button>' : ''}
            </div>
          </div>
        </div>
      </div>
    </div>
  `;
}

function setupPanelListeners(type, data) {
  if (type === 'login') {
    const form = document.getElementById('ea-login-form');
    form?.addEventListener('submit', async (e) => {
      e.preventDefault();
      const username = document.getElementById('ea-username').value;
      const password = document.getElementById('ea-password').value;
      const errorEl = document.querySelector('.ea-login-error');

      const response = await chrome.runtime.sendMessage({
        action: 'login',
        data: { username, password }
      });

      if (response.success) {
        document.getElementById('ea-panel').remove();
        showPanel('generate', currentJobData);
      } else {
        errorEl.textContent = response.error || 'Login failed';
        errorEl.style.display = 'block';
      }
    });
  } else if (type === 'generate') {
    const generateBtn = document.getElementById('ea-generate-btn');
    generateBtn?.addEventListener('click', async () => {
      const loading = document.getElementById('ea-loading');
      generateBtn.style.display = 'none';
      loading.style.display = 'block';

      // Progress animation helper
      const updateProgress = (step, percent, text) => {
        const fill = document.getElementById('ea-progress-fill');
        const progressText = document.getElementById('ea-progress-text');
        if (fill) fill.style.width = `${percent}%`;
        if (progressText) progressText.textContent = text;

        // Highlight current step
        for (let i = 1; i <= 4; i++) {
          const stepEl = document.getElementById(`step-${i}`);
          if (stepEl) {
            if (i < step) stepEl.style.color = '#22c55e';
            else if (i === step) stepEl.style.color = '#fff';
            else stepEl.style.color = '#6b7280';
          }
        }
      };

      // Get user's profile data from storage
      const profileData = await chrome.storage.local.get(['masterResume', 'additionalInfo', 'settings']);
      const userComments = document.getElementById('ea-user-comments')?.value || '';
      const preferredStyle = profileData.settings?.preferredStyle || 'professional-modern';
      const preferredFormat = profileData.settings?.preferredFormat || 'pdf';

      // Start progress animation
      updateProgress(1, 10, 'Analyzing job details...');

      // Simulate progress updates during API call
      const progressInterval = setInterval(() => {
        const fill = document.getElementById('ea-progress-fill');
        if (fill) {
          const current = parseInt(fill.style.width) || 0;
          if (current < 25) {
            updateProgress(1, current + 5, 'Analyzing job details...');
          } else if (current < 50) {
            updateProgress(2, current + 3, 'Generating tailored resume...');
          } else if (current < 75) {
            updateProgress(3, current + 2, 'Creating personalized cover letter...');
          } else if (current < 90) {
            updateProgress(4, current + 1, 'Finalizing documents...');
          }
        }
      }, 400);

      const response = await chrome.runtime.sendMessage({
        action: 'generate',
        data: {
          title: data.title,
          company: data.company,
          description: data.description,
          location: data.location,
          jobUrl: data.jobUrl,
          masterResume: profileData.masterResume || '',
          additionalInfo: profileData.additionalInfo || '',
          userComments: userComments
        }
      });

      clearInterval(progressInterval);

      if (response.success) {
        updateProgress(4, 100, '✅ Complete!');
        await new Promise(r => setTimeout(r, 300)); // Brief pause to show 100%
        document.getElementById('ea-panel').remove();
        showPanel('result', { ...response, hasEasyApply: data.hasEasyApply || data.hasDirectApply, preferredStyle, preferredFormat });
      } else {
        loading.innerHTML = `<p style="color:#ef4444;">Error: ${response.error}</p>`;
      }
    });
  } else if (type === 'result') {
    // Tab switching
    document.querySelectorAll('.ea-tab').forEach(tab => {
      tab.addEventListener('click', () => {
        document.querySelectorAll('.ea-tab').forEach(t => t.classList.remove('active'));
        document.querySelectorAll('.ea-tab-content').forEach(c => c.style.display = 'none');
        tab.classList.add('active');
        document.getElementById(`tab-${tab.dataset.tab}`).style.display = 'block';
      });
    });

    // Copy button
    document.getElementById('ea-copy')?.addEventListener('click', () => {
      const activeTab = document.querySelector('.ea-tab.active').dataset.tab;
      const content = document.querySelector(`#tab-${activeTab} pre`).textContent;
      navigator.clipboard.writeText(content);
      showNotification('Copied to clipboard!');
    });

    // Download button - uses selected format
    document.getElementById('ea-download')?.addEventListener('click', () => {
      const format = document.getElementById('ea-doc-format')?.value || 'pdf';
      downloadDocument(format, data);
    });

    // Auto-Fill button - the main feature!
    document.getElementById('ea-autofill')?.addEventListener('click', async () => {
      const btn = document.getElementById('ea-autofill');
      btn.textContent = '⏳ Filling...';
      btn.disabled = true;

      try {
        // Store generated content for auto-fill
        await chrome.runtime.sendMessage({
          action: 'updateUserProfile',
          data: {
            coverLetter: data.coverLetter,
            generatedResume: data.resume
          }
        });

        // Close the panel
        document.getElementById('ea-panel')?.remove();

        // Run auto-fill based on site
        const result = await runAutoFill(data);

        if (result.success) {
          showNotification(`✅ Filled ${result.fieldsFilledTotal || 'all'} fields! Review and submit.`);
        } else {
          showNotification(`⚠️ ${result.message || 'Some fields may need attention'}`);
        }
      } catch (error) {
        showNotification(`❌ Error: ${error.message}`);
      }
    });

    // Ask AI - Generate answer for custom question
    document.getElementById('ea-generate-answer')?.addEventListener('click', async () => {
      const questionInput = document.getElementById('ea-question-input');
      const question = questionInput.value.trim();

      if (!question) {
        showNotification('Please enter a question first');
        return;
      }

      const loadingEl = document.getElementById('ea-answer-loading');
      const resultEl = document.getElementById('ea-answer-result');
      const answerTextEl = document.getElementById('ea-answer-text');
      const btn = document.getElementById('ea-generate-answer');

      btn.disabled = true;
      btn.textContent = '⏳ Generating...';
      loadingEl.style.display = 'block';
      resultEl.style.display = 'none';

      try {
        // Call API to generate custom answer
        const response = await chrome.runtime.sendMessage({
          action: 'generateCustomAnswer',
          data: {
            question: question,
            jobTitle: data.title || currentJobData?.title,
            company: data.company || currentJobData?.company,
            jobDescription: data.description || currentJobData?.description
          }
        });

        if (response.success && response.answer) {
          answerTextEl.textContent = response.answer;
          resultEl.style.display = 'block';
        } else {
          showNotification(`Failed: ${response.error || 'Could not generate answer'}`);
        }
      } catch (error) {
        showNotification(`Error: ${error.message}`);
      } finally {
        btn.disabled = false;
        btn.textContent = '✨ Generate Answer';
        loadingEl.style.display = 'none';
      }
    });

    // Copy answer button
    document.getElementById('ea-copy-answer')?.addEventListener('click', () => {
      const answerText = document.getElementById('ea-answer-text').textContent;
      navigator.clipboard.writeText(answerText);
      showNotification('Answer copied to clipboard!');
    });
  }
}

// ==== AUTO-FILL IMPLEMENTATION ====

async function runAutoFill(generatedData) {
  const hostname = window.location.hostname;

  // Get user profile
  const profileResponse = await chrome.runtime.sendMessage({ action: 'getUserProfile' });
  const profile = profileResponse.profile || {};

  // Merge generated content into profile
  profile.coverLetter = generatedData.coverLetter;
  profile.generatedResume = generatedData.resume;

  // Site-specific auto-fill
  if (hostname.includes('linkedin.com')) {
    return await autoFillLinkedIn(profile, generatedData);
  } else if (hostname.includes('indeed.com')) {
    return await autoFillIndeed(profile, generatedData);
  } else {
    return await autoFillGeneric(profile, generatedData);
  }
}

// LinkedIn Easy Apply Auto-Fill
async function autoFillLinkedIn(profile, generatedData) {
  const result = { success: false, fieldsFilledTotal: 0, stepsCompleted: 0, message: '' };

  // Click Easy Apply button
  const applyBtn = document.querySelector('.jobs-apply-button--top-card, [data-control-name="jobdetails_topcard_inapply"]');
  if (applyBtn) {
    applyBtn.click();
    await wait(1500);
  }

  // Wait for modal
  let modal = null;
  for (let i = 0; i < 10; i++) {
    modal = document.querySelector('.jobs-easy-apply-modal, .artdeco-modal');
    if (modal) break;
    await wait(300);
  }

  if (!modal) {
    result.message = 'Easy Apply modal not found';
    return result;
  }

  // Process up to 10 steps
  let maxSteps = 10;
  while (maxSteps-- > 0) {
    const filled = await fillLinkedInStep(profile);
    result.fieldsFilledTotal += filled;
    result.stepsCompleted++;

    // Check if on review/submit step
    const submitBtn = document.querySelector('button[aria-label="Submit application"]');
    if (submitBtn) {
      result.success = true;
      result.message = 'Application ready for review. Click Submit to apply.';
      break;
    }

    // Try to click Next
    const nextBtn = document.querySelector('button[aria-label="Continue to next step"], button[aria-label="Next"]');
    if (nextBtn && !nextBtn.disabled) {
      nextBtn.click();
      await wait(1000);
    } else {
      // Try Review button
      const reviewBtn = document.querySelector('button[aria-label="Review your application"]');
      if (reviewBtn) {
        reviewBtn.click();
        await wait(1000);
      } else {
        result.success = true;
        result.message = 'All fields filled. Navigate and submit manually if needed.';
        break;
      }
    }
  }

  return result;
}

async function fillLinkedInStep(profile) {
  let filled = 0;

  // Common field fills
  filled += fillInput('input[name="firstName"], input[aria-label*="First name"]', profile.firstName);
  filled += fillInput('input[name="lastName"], input[aria-label*="Last name"]', profile.lastName);
  filled += fillInput('input[name="email"], input[type="email"]', profile.email);
  filled += fillInput('input[name="phone"], input[type="tel"], input[aria-label*="Phone"]', profile.phone);
  filled += fillInput('input[aria-label*="City"]', profile.city);
  filled += fillInput('input[aria-label*="LinkedIn"]', profile.linkedinUrl);
  filled += fillInput('input[aria-label*="Website"], input[aria-label*="portfolio"]', profile.portfolioUrl);

  // Cover letter / additional info
  filled += fillTextarea('textarea[aria-label*="cover"], textarea[aria-label*="additional"]', profile.coverLetter);

  // Handle dropdowns
  filled += fillSelect('select[aria-label*="Country"]', profile.country || 'United States');
  filled += fillSelect('select[aria-label*="State"]', profile.state);

  // Answer yes/no questions
  await answerLinkedInQuestions(profile);

  return filled;
}

async function answerLinkedInQuestions(profile) {
  const questionGroups = document.querySelectorAll('.jobs-easy-apply-form-section__grouping, fieldset');

  for (const group of questionGroups) {
    const text = group.textContent.toLowerCase();

    // Work authorization
    if (text.includes('authorized to work') || text.includes('legally authorized')) {
      clickRadioByText(group, profile.workAuthorized !== false ? 'yes' : 'no');
    }
    // Sponsorship
    else if (text.includes('sponsorship') || text.includes('visa')) {
      clickRadioByText(group, profile.needsSponsorship ? 'yes' : 'no');
    }
    // Relocation
    else if (text.includes('relocate')) {
      clickRadioByText(group, profile.willingToRelocate ? 'yes' : 'no');
    }
    // Background check
    else if (text.includes('background check')) {
      clickRadioByText(group, 'yes');
    }
    // Over 18
    else if (text.includes('18 years')) {
      clickRadioByText(group, 'yes');
    }
    // Experience years (numeric)
    else if (text.includes('years of experience')) {
      const input = group.querySelector('input[type="text"], input[type="number"]');
      if (input && !input.value && profile.yearsExperience) {
        input.value = profile.yearsExperience.toString();
        input.dispatchEvent(new Event('input', { bubbles: true }));
      }
    }
  }
}

// Indeed Apply Auto-Fill
async function autoFillIndeed(profile, generatedData) {
  const result = { success: false, fieldsFilledTotal: 0, stepsCompleted: 0, message: '' };

  // Click Apply button
  const applyBtn = document.querySelector('#indeedApplyButton, [data-testid="indeedApply-button"]');
  if (applyBtn) {
    applyBtn.click();
    await wait(2000);
  }

  // Process steps
  let maxSteps = 15;
  while (maxSteps-- > 0) {
    // Get document (may be in iframe)
    let doc = document;
    const iframe = document.querySelector('iframe[title*="Apply"], iframe[id*="indeed"]');
    if (iframe && iframe.contentDocument) {
      doc = iframe.contentDocument;
    }

    const filled = await fillIndeedStep(profile, doc);
    result.fieldsFilledTotal += filled;
    result.stepsCompleted++;

    // Check for submit
    const submitBtn = doc.querySelector('[data-testid="ia-submitButton"], button:contains("Submit")');
    if (submitBtn) {
      result.success = true;
      result.message = 'Application ready. Review and submit.';
      break;
    }

    // Try Continue
    const contBtn = doc.querySelector('[data-testid="ia-continueButton"], button[type="submit"]');
    if (contBtn && !contBtn.disabled) {
      contBtn.click();
      await wait(1500);
    } else {
      result.success = true;
      result.message = 'All available fields filled.';
      break;
    }
  }

  return result;
}

async function fillIndeedStep(profile, doc = document) {
  let filled = 0;

  filled += fillInput('input[name="firstName"], input[id*="firstName"]', profile.firstName, doc);
  filled += fillInput('input[name="lastName"], input[id*="lastName"]', profile.lastName, doc);
  filled += fillInput('input[name="name"], input[name="fullName"]', `${profile.firstName} ${profile.lastName}`, doc);
  filled += fillInput('input[name="email"], input[type="email"]', profile.email, doc);
  filled += fillInput('input[name="phone"], input[type="tel"]', profile.phone, doc);
  filled += fillInput('input[name="city"]', profile.city, doc);
  filled += fillInput('input[name="zip"], input[name="postalCode"]', profile.zip, doc);

  filled += fillTextarea('textarea[name="coverLetter"], textarea[id*="cover"]', profile.coverLetter, doc);
  filled += fillTextarea('textarea[name="message"]', profile.coverLetter, doc);

  filled += fillSelect('select[name="state"]', profile.state, doc);
  filled += fillSelect('select[name="country"]', profile.country || 'United States', doc);

  // Answer Indeed screening questions
  await answerIndeedQuestions(profile, doc);

  return filled;
}

async function answerIndeedQuestions(profile, doc = document) {
  const questions = doc.querySelectorAll('.ia-Question, [data-testid*="question"], fieldset');

  for (const q of questions) {
    const text = q.textContent.toLowerCase();

    if (text.includes('authorized to work') || text.includes('eligible to work')) {
      clickRadioByText(q, profile.workAuthorized !== false ? 'yes' : 'no');
    }
    else if (text.includes('sponsorship')) {
      clickRadioByText(q, profile.needsSponsorship ? 'yes' : 'no');
    }
    else if (text.includes('background check')) {
      clickRadioByText(q, 'yes');
    }
    else if (text.includes('18 years') || text.includes('at least 18')) {
      clickRadioByText(q, 'yes');
    }
    else if (text.includes('driver') && text.includes('license')) {
      clickRadioByText(q, profile.hasDriversLicense !== false ? 'yes' : 'no');
    }
    else if (text.includes('high school') || text.includes('diploma')) {
      clickRadioByText(q, 'yes');
    }
  }
}

// Generic Auto-Fill
async function autoFillGeneric(profile, generatedData) {
  const result = { success: false, fieldsFilledTotal: 0, message: '' };

  // Fill common fields
  result.fieldsFilledTotal += fillInput('input[name="firstName"], input[name="first_name"]', profile.firstName);
  result.fieldsFilledTotal += fillInput('input[name="lastName"], input[name="last_name"]', profile.lastName);
  result.fieldsFilledTotal += fillInput('input[name="email"], input[type="email"]', profile.email);
  result.fieldsFilledTotal += fillInput('input[name="phone"], input[type="tel"]', profile.phone);
  result.fieldsFilledTotal += fillInput('input[name="city"]', profile.city);
  result.fieldsFilledTotal += fillInput('input[name="linkedin"]', profile.linkedinUrl);
  result.fieldsFilledTotal += fillTextarea('textarea[name="coverLetter"], textarea[name="cover_letter"]', profile.coverLetter);

  result.success = result.fieldsFilledTotal > 0;
  result.message = result.success ? `Filled ${result.fieldsFilledTotal} fields` : 'No matching fields found';

  return result;
}

// ==== HELPER FUNCTIONS ====

function fillInput(selector, value, doc = document) {
  if (!value) return 0;
  const el = doc.querySelector(selector);
  if (el && el.offsetParent !== null && !el.value) {
    el.value = value;
    el.dispatchEvent(new Event('input', { bubbles: true }));
    el.dispatchEvent(new Event('change', { bubbles: true }));
    return 1;
  }
  return 0;
}

function fillTextarea(selector, value, doc = document) {
  if (!value) return 0;
  const el = doc.querySelector(selector);
  if (el && el.offsetParent !== null && !el.value) {
    el.value = value;
    el.dispatchEvent(new Event('input', { bubbles: true }));
    return 1;
  }
  return 0;
}

function fillSelect(selector, value, doc = document) {
  if (!value) return 0;
  const el = doc.querySelector(selector);
  if (el && el.offsetParent !== null) {
    for (const opt of el.options) {
      if (opt.text.toLowerCase().includes(value.toLowerCase()) || opt.value.toLowerCase() === value.toLowerCase()) {
        el.value = opt.value;
        el.dispatchEvent(new Event('change', { bubbles: true }));
        return 1;
      }
    }
  }
  return 0;
}

function clickRadioByText(container, targetText) {
  // Try value match
  const radio = container.querySelector(`input[type="radio"][value="${targetText}"], input[type="radio"][value="${targetText.charAt(0).toUpperCase() + targetText.slice(1)}"]`);
  if (radio) {
    radio.click();
    return true;
  }

  // Try label text match
  const labels = container.querySelectorAll('label');
  for (const label of labels) {
    if (label.textContent.toLowerCase().trim() === targetText || label.textContent.toLowerCase().includes(targetText)) {
      label.click();
      return true;
    }
  }
  return false;
}

function wait(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

async function downloadDocument(format, data) {
  const activeTab = document.querySelector('.ea-tab.active').dataset.tab;
  const type = activeTab === 'cover' ? 'coverLetter' : activeTab === 'email' ? 'emailDraft' : 'resume';
  const content = data[type === 'coverLetter' ? 'coverLetter' : type === 'emailDraft' ? 'emailDraft' : 'resume'];
  const style = document.getElementById('ea-doc-style')?.value || 'professional-modern';

  const response = await chrome.runtime.sendMessage({
    action: 'download',
    data: { type, content, format, style }
  });

  if (response.success) {
    // Create download link
    const a = document.createElement('a');
    a.href = response.data;
    a.download = response.filename;
    a.click();
  }
}

function showNotification(message) {
  const notif = document.createElement('div');
  notif.className = 'ea-notification';
  notif.textContent = message;
  document.body.appendChild(notif);
  setTimeout(() => notif.remove(), 2000);
}

// Initialize detection with retries
// LinkedIn and other SPAs often load content dynamically
let detectionRetries = 0;
const MAX_RETRIES = 10;

function runDetectionWithRetries() {
  detectJob();

  // Retry periodically for a few seconds to catch lazy-loaded content
  const retryInterval = setInterval(() => {
    detectionRetries++;
    const fabExists = document.getElementById('ea-fab');

    // If we found a job and injected FAB, stop retrying
    if (fabExists) {
      clearInterval(retryInterval);
      return;
    }

    detectJob();

    if (detectionRetries >= MAX_RETRIES) {
      clearInterval(retryInterval);
    }
  }, 1000);
}

// Initial run
runDetectionWithRetries();

// Watch for page changes (SPA navigation)
let lastUrl = location.href;
new MutationObserver(() => {
  if (location.href !== lastUrl) {
    lastUrl = location.href;
    console.log('AutoJob: URL changed', lastUrl);
    detectionRetries = 0; // Reset retries on navigation

    // Clear existing FAB to force re-detection
    const oldFab = document.getElementById('ea-fab');
    if (oldFab) oldFab.remove();

    runDetectionWithRetries();
  }
}).observe(document, { subtree: true, childList: true });

// Also detect on page load complete and historical state changes
window.addEventListener('load', () => setTimeout(runDetectionWithRetries, 1500));
window.addEventListener('popstate', () => setTimeout(runDetectionWithRetries, 500));

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'getJobData') {
    sendResponse(currentJobData);
    return true; // Keep channel open for async response if needed (though we reply synchronously here)
  }

  // Analyze page for fillable fields
  if (request.action === 'analyzePage') {
    if (window.PageAnalyzer) {
      const results = window.PageAnalyzer.analyze();
      sendResponse({ success: true, results });
    } else {
      sendResponse({ success: false, error: 'PageAnalyzer not loaded' });
    }
    return true;
  }

  // Autofill the current page
  if (request.action === 'autofillPage') {
    (async () => {
      try {
        if (!window.PageAnalyzer || !window.SmartAutofill) {
          sendResponse({ success: false, error: 'Autofill modules not loaded' });
          return;
        }

        const analysisResults = window.PageAnalyzer.analyze();
        const fillResult = await window.SmartAutofill.fillPage(analysisResults);

        // Check for resume upload
        if (analysisResults.hasResumeUpload) {
          // Show resume generation prompt
          showResumePrompt(analysisResults.resumeUploadElement);
        }

        sendResponse(fillResult);
      } catch (e) {
        console.error('[Eric\'s AutoApply] Autofill error:', e);
        sendResponse({ success: false, error: e.message });
      }
    })();
    return true;
  }

  // Check for resume upload on page
  if (request.action === 'checkResumeUpload') {
    if (window.PageAnalyzer) {
      const result = window.PageAnalyzer.detectResumeUpload();
      sendResponse(result);
    } else {
      sendResponse({ found: false });
    }
    return true;
  }

  // Force detect - bypasses normal checks and tries to extract data
  if (request.action === 'forceDetect') {
    console.log("[Eric's AutoApply] Force detect triggered");

    // Try all extractors to find job data
    const extractor = getExtractor();
    currentJobData = extractor.extract();

    // If generic extractor didn't find much, try harder
    if (!currentJobData.title || currentJobData.title === '') {
      currentJobData = extractGenericForced();
    }

    if (currentJobData.title) {
      console.log("[Eric's AutoApply] Force detect found:", currentJobData.title);
      injectFAB();
      sendResponse({ success: true, title: currentJobData.title, data: currentJobData });
    } else {
      console.log("[Eric's AutoApply] Force detect: no job data found");
      sendResponse({ success: false });
    }
    return true;
  }

  if (request.action === 'openQuickApply') {
    if (fabInjected) {
      const fab = document.getElementById('ea-fab');
      if (fab) {
        fab.querySelector('.ea-fab-button').click();
      }
    } else {
      detectJob(); // Try to detect again
      setTimeout(() => {
        const fab = document.getElementById('ea-fab');
        if (fab) fab.querySelector('.ea-fab-button').click();
      }, 500);
    }
  }
});

// Show prompt when resume upload is detected
function showResumePrompt(fileInput) {
  // Don't show if already shown
  if (document.getElementById('ea-resume-prompt')) return;

  const prompt = document.createElement('div');
  prompt.id = 'ea-resume-prompt';
  prompt.innerHTML = `
    <div class="ea-resume-prompt-content">
      <span class="ea-resume-icon">📄</span>
      <span class="ea-resume-text">Resume upload detected</span>
      <button class="ea-resume-btn">Generate Tailored Resume</button>
      <button class="ea-resume-close">×</button>
    </div>
  `;

  prompt.querySelector('.ea-resume-btn').addEventListener('click', async () => {
    prompt.remove();
    // Trigger document generation
    const authResponse = await chrome.runtime.sendMessage({ action: 'checkAuth' });
    if (authResponse.isAuthenticated) {
      // Extract any available job context
      const extractor = getExtractor();
      let jobContext = extractor.extract();
      if (!jobContext.title) {
        jobContext = extractGenericForced();
      }
      showPanel('generate', { ...jobContext, resumeUploadElement: fileInput });
    } else {
      showPanel('login');
    }
  });

  prompt.querySelector('.ea-resume-close').addEventListener('click', () => {
    prompt.remove();
  });

  document.body.appendChild(prompt);

  // Auto-dismiss after 10 seconds
  setTimeout(() => {
    prompt.remove();
  }, 10000);
}

// Auto-analyze page on load for forms (runs after job detection)
async function runFormAnalysis() {
  if (!window.PageAnalyzer) return;

  // Check settings for auto-fill preference
  try {
    const response = await chrome.runtime.sendMessage({ action: 'getSettings' });
    if (response.success && response.settings.autoFillEnabled) {
      const results = window.PageAnalyzer.analyze();

      if (results.totalFillableFields > 0) {
        console.log(`[Eric's AutoApply] Found ${results.totalFillableFields} fillable fields`);

        // Check for resume upload
        if (results.hasResumeUpload) {
          showResumePrompt(results.resumeUploadElement);
        }
      }
    }
  } catch (e) {
    // Settings check failed, skip auto-analysis
  }
}

// Run form analysis after a short delay
setTimeout(runFormAnalysis, 2000);

// Forced extraction - tries harder to find job info on any page
function extractGenericForced() {
  // Look for the most prominent heading as job title
  const headings = document.querySelectorAll('h1, h2');
  let title = '';
  for (const h of headings) {
    const text = h.textContent.trim();
    if (text.length > 5 && text.length < 150) {
      title = text;
      break;
    }
  }

  // Look for company name in common places
  let company = '';
  const companySelectors = [
    '[class*="company"]', '[id*="company"]',
    '[class*="employer"]', '[id*="employer"]',
    '[class*="organization"]',
    'meta[property="og:site_name"]'
  ];
  for (const sel of companySelectors) {
    const el = document.querySelector(sel);
    if (el) {
      company = el.content || el.textContent?.trim() || '';
      if (company && company.length < 100) break;
    }
  }

  // Get page description as job description
  let description = '';
  const metaDesc = document.querySelector('meta[name="description"]');
  if (metaDesc) description = metaDesc.content;

  // Or look for main content
  if (!description) {
    const mainContent = document.querySelector('main, article, [role="main"], .content, #content');
    if (mainContent) {
      description = mainContent.textContent.substring(0, 2000);
    }
  }

  return {
    success: !!title,
    site: 'generic',
    title: title,
    company: company,
    location: '',
    description: description,
    jobUrl: window.location.href
  };
}
