// Job Extractor Module
// Provides unified interface for extracting job data from any supported site

// This file is loaded alongside detector.js but provides the full extraction logic
// The detector.js has simplified inline versions for quick detection

class JobExtractor {
    constructor() {
        this.currentJobData = null;
    }

    // Get the appropriate extractor for the current site
    getSiteExtractor() {
        const url = window.location.href;

        if (url.includes('linkedin.com')) {
            return new LinkedInJobExtractor();
        } else if (url.includes('indeed.com')) {
            return new IndeedJobExtractor();
        } else if (url.includes('glassdoor.com')) {
            return new GlassdoorJobExtractor();
        } else if (url.includes('ziprecruiter.com')) {
            return new ZipRecruiterJobExtractor();
        } else {
            return new GenericJobExtractor();
        }
    }

    // Extract job data from current page
    extract() {
        const extractor = this.getSiteExtractor();
        this.currentJobData = extractor.extract();
        return this.currentJobData;
    }

    // Check if we can auto-apply on this site
    canAutoApply() {
        if (!this.currentJobData) return false;
        return this.currentJobData.hasEasyApply || this.currentJobData.hasDirectApply;
    }
}

// LinkedIn Extractor
class LinkedInJobExtractor {
    extract() {
        const selectors = {
            title: [
                '.job-details-jobs-unified-top-card__job-title',
                '.jobs-unified-top-card__job-title',
                'h1.t-24',
                '.job-details h1'
            ],
            company: [
                '.job-details-jobs-unified-top-card__company-name',
                '.jobs-unified-top-card__company-name',
                '.job-details-jobs-unified-top-card__primary-description-container a'
            ],
            location: [
                '.job-details-jobs-unified-top-card__bullet',
                '.jobs-unified-top-card__bullet'
            ],
            description: [
                '.jobs-description-content__text',
                '.jobs-description__content',
                '#job-details'
            ]
        };

        return {
            success: true,
            site: 'linkedin',
            title: this.findText(selectors.title),
            company: this.findText(selectors.company),
            location: this.findText(selectors.location),
            description: this.findText(selectors.description),
            salary: this.extractSalary(),
            hasEasyApply: this.hasEasyApply(),
            jobUrl: window.location.href,
            jobId: this.extractJobId()
        };
    }

    findText(selectors) {
        for (const sel of selectors) {
            const el = document.querySelector(sel);
            if (el) return el.textContent.trim();
        }
        return '';
    }

    extractSalary() {
        const salaryEl = document.querySelector('.job-details-jobs-unified-top-card__job-insight');
        if (salaryEl && salaryEl.textContent.includes('$')) {
            return salaryEl.textContent.trim();
        }
        return null;
    }

    hasEasyApply() {
        return !!document.querySelector('[data-control-name="jobdetails_topcard_inapply"], .jobs-apply-button--top-card, button[aria-label*="Easy Apply"]');
    }

    extractJobId() {
        const match = window.location.href.match(/\/jobs\/view\/(\d+)/);
        return match ? match[1] : null;
    }
}

// Indeed Extractor  
class IndeedJobExtractor {
    extract() {
        const selectors = {
            title: [
                '[data-testid="jobsearch-JobInfoHeader-title"]',
                '.jobsearch-JobInfoHeader-title',
                'h1.icl-u-xs-mb--xs'
            ],
            company: [
                '[data-testid="inlineHeader-companyName"]',
                '.jobsearch-InlineCompanyRating a'
            ],
            location: [
                '[data-testid="job-location"]',
                '[data-testid="jobsearch-JobInfoHeader-companyLocation"]'
            ],
            description: [
                '#jobDescriptionText',
                '.jobsearch-jobDescriptionText'
            ]
        };

        return {
            success: true,
            site: 'indeed',
            title: this.findText(selectors.title),
            company: this.findText(selectors.company),
            location: this.findText(selectors.location),
            description: this.findText(selectors.description),
            salary: this.extractSalary(),
            hasDirectApply: !!document.querySelector('#indeedApplyButton'),
            jobUrl: window.location.href,
            jobId: this.extractJobId()
        };
    }

    findText(selectors) {
        for (const sel of selectors) {
            const el = document.querySelector(sel);
            if (el) return el.textContent.trim();
        }
        return '';
    }

    extractSalary() {
        const el = document.querySelector('[data-testid="jobsearch-JobInfoHeader-salary"], #salaryInfoAndJobType');
        return el ? el.textContent.trim() : null;
    }

    extractJobId() {
        const match = window.location.href.match(/jk=([a-f0-9]+)/i);
        return match ? match[1] : null;
    }
}

// Glassdoor Extractor
class GlassdoorJobExtractor {
    extract() {
        return {
            success: true,
            site: 'glassdoor',
            title: document.querySelector('[data-test="job-title"]')?.textContent.trim() || '',
            company: document.querySelector('[data-test="employer-name"]')?.textContent.trim() || '',
            location: document.querySelector('[data-test="location"]')?.textContent.trim() || '',
            description: document.querySelector('.jobDescriptionContent')?.textContent.trim() || '',
            hasDirectApply: !!document.querySelector('.gd-apply-button'),
            jobUrl: window.location.href
        };
    }
}

// ZipRecruiter Extractor
class ZipRecruiterJobExtractor {
    extract() {
        return {
            success: true,
            site: 'ziprecruiter',
            title: document.querySelector('.job_title, h1.title')?.textContent.trim() || '',
            company: document.querySelector('.hiring_company, .company_name')?.textContent.trim() || '',
            location: document.querySelector('.location, .job_location')?.textContent.trim() || '',
            description: document.querySelector('.job_description, #job_desc')?.textContent.trim() || '',
            hasDirectApply: !!document.querySelector('.apply_button, #quick_apply'),
            jobUrl: window.location.href
        };
    }
}

// Generic Extractor for any career page
class GenericJobExtractor {
    extract() {
        return {
            success: true,
            site: 'generic',
            title: this.findTitle(),
            company: this.findCompany(),
            location: this.findLocation(),
            description: this.findDescription(),
            hasDirectApply: false,
            jobUrl: window.location.href,
            confidence: this.calculateConfidence()
        };
    }

    findTitle() {
        const h1 = document.querySelector('h1');
        if (h1) return h1.textContent.trim();

        const titleEl = document.querySelector('[class*="title"], [class*="job-title"]');
        if (titleEl) return titleEl.textContent.trim();

        return document.title.split('|')[0].split('-')[0].trim();
    }

    findCompany() {
        const companyEl = document.querySelector('[class*="company"], [class*="employer"]');
        if (companyEl) return companyEl.textContent.trim();

        // Try to extract from domain
        return window.location.hostname.replace('www.', '').split('.')[0];
    }

    findLocation() {
        const locationEl = document.querySelector('[class*="location"]');
        return locationEl ? locationEl.textContent.trim() : 'Location not specified';
    }

    findDescription() {
        const descEl = document.querySelector('[class*="description"], article, main');
        return descEl ? descEl.textContent.trim().slice(0, 5000) : '';
    }

    calculateConfidence() {
        let score = 0;
        if (this.findTitle()) score += 30;
        if (this.findDescription().length > 200) score += 40;
        if (document.body.textContent.toLowerCase().includes('apply')) score += 20;
        if (document.body.textContent.toLowerCase().includes('responsibilities')) score += 10;
        return Math.min(score, 100);
    }
}

// Export for use in detector.js
window.JobExtractor = JobExtractor;
