// Page Analyzer - Detects and classifies form fields on any page
// Part of the exploit.team AutoJob extension

const PageAnalyzer = {
    // Field type patterns for classification
    patterns: {
        firstName: {
            attributes: ['first_name', 'firstname', 'fname', 'given_name', 'givenname', 'first-name'],
            labels: ['first name', 'given name', 'forename'],
            autocomplete: ['given-name']
        },
        lastName: {
            attributes: ['last_name', 'lastname', 'lname', 'family_name', 'familyname', 'surname', 'last-name'],
            labels: ['last name', 'family name', 'surname'],
            autocomplete: ['family-name']
        },
        fullName: {
            attributes: ['full_name', 'fullname', 'name', 'your_name', 'applicant_name'],
            labels: ['full name', 'your name', 'name'],
            autocomplete: ['name']
        },
        email: {
            attributes: ['email', 'e-mail', 'user_email', 'emailaddress', 'email_address'],
            labels: ['email', 'e-mail', 'email address'],
            autocomplete: ['email'],
            types: ['email']
        },
        phone: {
            attributes: ['phone', 'telephone', 'mobile', 'cell', 'tel', 'phone_number', 'phonenumber'],
            labels: ['phone', 'telephone', 'mobile', 'cell phone'],
            autocomplete: ['tel', 'tel-national'],
            types: ['tel']
        },
        address1: {
            attributes: ['address', 'address1', 'street', 'street_address', 'line1', 'address_line_1'],
            labels: ['address', 'street address', 'address line 1'],
            autocomplete: ['street-address', 'address-line1']
        },
        address2: {
            attributes: ['address2', 'apt', 'suite', 'unit', 'line2', 'address_line_2', 'apartment'],
            labels: ['address line 2', 'apt', 'suite', 'unit', 'apartment'],
            autocomplete: ['address-line2']
        },
        city: {
            attributes: ['city', 'locality', 'town'],
            labels: ['city', 'town', 'locality'],
            autocomplete: ['address-level2']
        },
        state: {
            attributes: ['state', 'region', 'province', 'state_province'],
            labels: ['state', 'province', 'region'],
            autocomplete: ['address-level1']
        },
        zip: {
            attributes: ['zip', 'postal', 'postcode', 'postal_code', 'zipcode', 'zip_code'],
            labels: ['zip', 'postal code', 'postcode', 'zip code'],
            autocomplete: ['postal-code']
        },
        country: {
            attributes: ['country', 'nation', 'country_code'],
            labels: ['country', 'nation'],
            autocomplete: ['country', 'country-name']
        },
        linkedin: {
            attributes: ['linkedin', 'linkedin_url', 'linkedinurl'],
            labels: ['linkedin', 'linkedin url', 'linkedin profile']
        },
        portfolio: {
            attributes: ['website', 'portfolio', 'url', 'personal_website', 'homepage'],
            labels: ['website', 'portfolio', 'personal website', 'homepage', 'url']
        },
        yearsExperience: {
            attributes: ['years_experience', 'experience', 'years_of_experience', 'work_experience'],
            labels: ['years of experience', 'experience', 'how many years']
        },
        resume: {
            attributes: ['resume', 'cv', 'curriculum', 'resume_upload'],
            labels: ['resume', 'cv', 'curriculum vitae', 'upload resume', 'upload cv'],
            isFile: true
        },
        customQuestion: {
            // Catch-all for textarea questions that don't match other patterns
            isTextarea: true,
            labels: ['tell us', 'describe', 'explain', 'why do you', 'additional', 'comments', 'notes', 'cover letter', 'message']
        }
    },

    // Fields to skip (security/sensitive)
    skipPatterns: ['password', 'captcha', 'credit', 'card', 'cvv', 'ssn', 'social_security'],

    /**
     * Quick scan - fast count of fillable fields without full classification
     * Use this for initial page detection
     */
    quickScan() {
        const inputs = document.querySelectorAll('input:not([type="hidden"]):not([type="submit"]):not([type="button"]):not([type="reset"]):not([disabled]), select:not([disabled]), textarea:not([disabled])');
        let count = 0;

        for (const input of inputs) {
            // Skip invisible elements
            if (input.offsetParent === null) continue;
            // Skip password and sensitive fields
            if (input.type === 'password') continue;
            const name = (input.name || input.id || '').toLowerCase();
            if (this.skipPatterns.some(p => name.includes(p))) continue;
            count++;
        }

        return count;
    },

    /**
     * Analyze all forms on the current page
     * Returns a map of detected fields with their types and elements
     */
    analyze() {

        const results = {
            forms: [],
            fields: [],
            hasResumeUpload: false,
            resumeUploadElement: null,
            totalFillableFields: 0
        };

        // Get all input-like elements (inside and outside forms)
        const allInputs = document.querySelectorAll('input, select, textarea');
        const processedElements = new Set();

        // Process forms first
        const forms = document.querySelectorAll('form');
        forms.forEach((form, formIndex) => {
            const formFields = [];
            const inputs = form.querySelectorAll('input, select, textarea');

            inputs.forEach(input => {
                if (processedElements.has(input)) return;
                processedElements.add(input);

                const fieldInfo = this.classifyField(input);
                if (fieldInfo) {
                    fieldInfo.formIndex = formIndex;
                    formFields.push(fieldInfo);
                    results.fields.push(fieldInfo);

                    if (fieldInfo.type === 'resume') {
                        results.hasResumeUpload = true;
                        results.resumeUploadElement = input;
                    }

                    if (!fieldInfo.skip) {
                        results.totalFillableFields++;
                    }
                }
            });

            results.forms.push({
                index: formIndex,
                element: form,
                fields: formFields,
                action: form.action || '',
                method: form.method || 'get'
            });
        });

        // Process standalone inputs (not in forms)
        allInputs.forEach(input => {
            if (processedElements.has(input)) return;
            processedElements.add(input);

            const fieldInfo = this.classifyField(input);
            if (fieldInfo) {
                fieldInfo.formIndex = -1; // Not in a form
                results.fields.push(fieldInfo);

                if (fieldInfo.type === 'resume') {
                    results.hasResumeUpload = true;
                    results.resumeUploadElement = input;
                }

                if (!fieldInfo.skip) {
                    results.totalFillableFields++;
                }
            }
        });

        console.log(`[PageAnalyzer] Found ${results.fields.length} fields, ${results.totalFillableFields} fillable`);
        return results;
    },

    /**
     * Classify a single field element
     */
    classifyField(element) {
        // Skip hidden, disabled, or readonly fields
        if (element.type === 'hidden' || element.disabled || element.readOnly) {
            return null;
        }

        // Skip submit/button types
        if (['submit', 'button', 'reset', 'image'].includes(element.type)) {
            return null;
        }

        const context = this.getFieldContext(element);
        const contextLower = context.toLowerCase();

        // Check skip patterns
        const shouldSkip = this.skipPatterns.some(pattern => contextLower.includes(pattern));

        // Get element attributes for matching
        const name = (element.name || '').toLowerCase();
        const id = (element.id || '').toLowerCase();
        const placeholder = (element.placeholder || '').toLowerCase();
        const autocomplete = (element.autocomplete || '').toLowerCase();
        const type = (element.type || 'text').toLowerCase();

        // Find matching field type
        let detectedType = 'unknown';
        let confidence = 0;

        for (const [fieldType, patterns] of Object.entries(this.patterns)) {
            let score = 0;

            // Check if this is a file input for resume
            if (patterns.isFile && type === 'file') {
                const resumeNearby = /resume|cv|curriculum|upload.*document/i.test(contextLower);
                if (resumeNearby) {
                    score += 10;
                }
            }

            // Check attributes
            if (patterns.attributes) {
                if (patterns.attributes.some(attr => name.includes(attr))) score += 3;
                if (patterns.attributes.some(attr => id.includes(attr))) score += 3;
                if (patterns.attributes.some(attr => placeholder.includes(attr))) score += 2;
            }

            // Check labels
            if (patterns.labels) {
                if (patterns.labels.some(label => contextLower.includes(label))) score += 4;
            }

            // Check autocomplete
            if (patterns.autocomplete) {
                if (patterns.autocomplete.some(ac => autocomplete.includes(ac))) score += 5;
            }

            // Check input types
            if (patterns.types && patterns.types.includes(type)) {
                score += 4;
            }

            if (score > confidence) {
                confidence = score;
                detectedType = fieldType;
            }
        }

        // Minimum confidence threshold
        if (confidence < 2) {
            detectedType = 'unknown';
        }

        return {
            element,
            type: detectedType,
            confidence,
            skip: shouldSkip,
            tagName: element.tagName.toLowerCase(),
            inputType: type,
            name: element.name,
            id: element.id,
            context: context.slice(0, 100) // Truncate for storage
        };
    },

    /**
     * Get context text for a field (label, nearby text, parent container)
     */
    getFieldContext(element) {
        const contextParts = [];

        // Get associated label via for attribute
        if (element.id) {
            const label = document.querySelector(`label[for="${element.id}"]`);
            if (label) {
                contextParts.push(label.textContent.trim());
            }
        }

        // Get parent label
        const parentLabel = element.closest('label');
        if (parentLabel) {
            contextParts.push(parentLabel.textContent.trim());
        }

        // Get placeholder
        if (element.placeholder) {
            contextParts.push(element.placeholder);
        }

        // Get aria-label
        if (element.getAttribute('aria-label')) {
            contextParts.push(element.getAttribute('aria-label'));
        }

        // Get nearby text from parent container
        const container = element.closest('div, fieldset, section, td');
        if (container) {
            // Get text nodes directly in container
            const walker = document.createTreeWalker(container, NodeFilter.SHOW_TEXT);
            let node;
            while (node = walker.nextNode()) {
                const text = node.textContent.trim();
                if (text && text.length < 100) {
                    contextParts.push(text);
                }
            }
        }

        return contextParts.join(' ');
    },

    /**
     * Detect if the page has a resume/CV upload field
     */
    detectResumeUpload() {
        const fileInputs = document.querySelectorAll('input[type="file"]');

        for (const input of fileInputs) {
            const context = this.getFieldContext(input);
            if (/resume|cv|curriculum|upload.*document/i.test(context)) {
                return {
                    found: true,
                    element: input,
                    context: context.slice(0, 200)
                };
            }
        }

        // Also check for drop zones
        const dropZones = document.querySelectorAll('[class*="drop"], [class*="upload"], [id*="drop"], [id*="upload"]');
        for (const zone of dropZones) {
            const text = zone.textContent.toLowerCase();
            if (/resume|cv|drag|drop/i.test(text) && text.length < 500) {
                return {
                    found: true,
                    element: zone,
                    context: text.slice(0, 200),
                    isDropZone: true
                };
            }
        }

        return { found: false };
    }
};

// Export for content script usage
window.PageAnalyzer = PageAnalyzer;
