// Smart Autofill Engine - Fills forms using user profile data
// Part of the exploit.team AutoJob extension

const SmartAutofill = {
    // Cache for user profile
    profile: null,

    /**
     * Initialize with user profile data
     */
    async init() {
        try {
            const response = await chrome.runtime.sendMessage({ action: 'getUserProfile' });
            if (response.success && response.profile) {
                this.profile = response.profile;
                console.log('[SmartAutofill] Profile loaded successfully');
                return true;
            }
        } catch (e) {
            console.error('[SmartAutofill] Failed to load profile:', e);
        }
        return false;
    },

    /**
     * Fill all fields on the page in one action
     * Convenience method that combines init + analyze + fillPage
     */
    async fillAll() {
        await this.init();
        if (!this.profile) {
            console.log('[SmartAutofill] No profile available');
            return { success: false, filled: 0, error: 'No profile configured. Set up your profile first.' };
        }

        if (!window.PageAnalyzer) {
            return { success: false, filled: 0, error: 'Page analyzer not loaded' };
        }

        const analysis = window.PageAnalyzer.analyze();
        return await this.fillPage(analysis);
    },

    /**
     * Get value for a field type from user profile
     */

    getValueForType(fieldType) {
        if (!this.profile) return null;

        const mappings = {
            firstName: this.profile.firstName,
            lastName: this.profile.lastName,
            fullName: `${this.profile.firstName || ''} ${this.profile.lastName || ''}`.trim(),
            email: this.profile.email,
            phone: this.profile.phone,
            address1: this.profile.address1,
            address2: this.profile.address2,
            city: this.profile.city,
            state: this.profile.state,
            zip: this.profile.zip,
            country: this.profile.country,
            linkedin: this.profile.linkedinUrl,
            portfolio: this.profile.portfolioUrl,
            yearsExperience: this.profile.yearsExperience?.toString()
        };

        return mappings[fieldType] || null;
    },

    /**
     * Fill a single field with the appropriate value
     */
    fillField(fieldInfo) {
        if (!fieldInfo || fieldInfo.skip || fieldInfo.type === 'unknown' || fieldInfo.type === 'resume') {
            return false;
        }

        const value = this.getValueForType(fieldInfo.type);
        if (!value) {
            return false;
        }

        const element = fieldInfo.element;

        // Already has value? Skip unless empty
        if (element.value && element.value.trim()) {
            return false;
        }

        try {
            if (element.tagName.toLowerCase() === 'select') {
                return this.fillSelect(element, value);
            } else if (element.tagName.toLowerCase() === 'textarea') {
                return this.fillTextarea(element, value);
            } else if (element.type === 'checkbox') {
                return this.fillCheckbox(element, fieldInfo.type);
            } else {
                return this.fillInput(element, value);
            }
        } catch (e) {
            console.error('[SmartAutofill] Error filling field:', e);
            return false;
        }
    },

    /**
     * Fill a text input field
     */
    fillInput(element, value) {
        // Set value using native setter (bypasses React/Vue)
        const nativeInputValueSetter = Object.getOwnPropertyDescriptor(
            window.HTMLInputElement.prototype, 'value'
        ).set;

        nativeInputValueSetter.call(element, value);

        // Dispatch events for framework compatibility
        element.dispatchEvent(new Event('input', { bubbles: true }));
        element.dispatchEvent(new Event('change', { bubbles: true }));
        element.dispatchEvent(new Event('blur', { bubbles: true }));

        // Visual feedback - subtle green flash
        this.flashField(element);

        return true;
    },

    /**
     * Fill a select dropdown
     */
    fillSelect(element, value) {
        const valueLower = value.toLowerCase();

        // Try to find matching option
        for (const option of element.options) {
            const optionText = option.text.toLowerCase();
            const optionValue = option.value.toLowerCase();

            if (optionText.includes(valueLower) || optionValue.includes(valueLower) ||
                valueLower.includes(optionText) || valueLower.includes(optionValue)) {
                element.value = option.value;
                element.dispatchEvent(new Event('change', { bubbles: true }));
                this.flashField(element);
                return true;
            }
        }

        return false;
    },

    /**
     * Fill a textarea
     */
    fillTextarea(element, value) {
        const nativeTextAreaValueSetter = Object.getOwnPropertyDescriptor(
            window.HTMLTextAreaElement.prototype, 'value'
        ).set;

        nativeTextAreaValueSetter.call(element, value);

        element.dispatchEvent(new Event('input', { bubbles: true }));
        element.dispatchEvent(new Event('change', { bubbles: true }));

        this.flashField(element);
        return true;
    },

    /**
     * Fill a checkbox based on profile preferences
     */
    fillCheckbox(element, fieldType) {
        const context = element.closest('div, fieldset, label')?.textContent?.toLowerCase() || '';

        // Work authorization
        if (/authorized|eligible|legal.*work/i.test(context)) {
            element.checked = this.profile.workAuthorized !== false;
            element.dispatchEvent(new Event('change', { bubbles: true }));
            return true;
        }

        // Sponsorship
        if (/sponsor|visa|h1b/i.test(context)) {
            element.checked = this.profile.needsSponsorship === true;
            element.dispatchEvent(new Event('change', { bubbles: true }));
            return true;
        }

        return false;
    },

    /**
     * Visual feedback - flash field with green border
     */
    flashField(element) {
        const originalBoxShadow = element.style.boxShadow;
        const originalTransition = element.style.transition;

        element.style.transition = 'box-shadow 0.3s ease';
        element.style.boxShadow = '0 0 0 2px #22c55e';

        setTimeout(() => {
            element.style.boxShadow = originalBoxShadow;
            setTimeout(() => {
                element.style.transition = originalTransition;
            }, 300);
        }, 1000);
    },

    /**
     * Fill all detected fields on the page
     */
    async fillPage(analysisResults) {
        if (!this.profile) {
            await this.init();
        }

        if (!this.profile) {
            console.log('[SmartAutofill] No profile available, cannot autofill');
            return { success: false, filled: 0, error: 'No profile configured' };
        }

        let filledCount = 0;
        const filledFields = [];

        for (const fieldInfo of analysisResults.fields) {
            if (this.fillField(fieldInfo)) {
                filledCount++;
                filledFields.push(fieldInfo.type);
            }
        }

        console.log(`[SmartAutofill] Filled ${filledCount} fields:`, filledFields);

        // Update badge
        if (filledCount > 0) {
            try {
                await chrome.runtime.sendMessage({
                    action: 'updateBadge',
                    data: { count: filledCount }
                });
            } catch (e) {
                // Badge update is optional
            }
        }

        return {
            success: true,
            filled: filledCount,
            filledFields
        };
    },

    /**
     * Handle yes/no screening questions based on profile
     */
    handleScreeningQuestion(container) {
        const text = container.textContent.toLowerCase();

        // Work authorization
        if (/authorized|eligible|legal.*work/i.test(text)) {
            return this.selectYesNo(container, this.profile.workAuthorized !== false);
        }

        // Sponsorship
        if (/sponsor|visa|h1b/i.test(text)) {
            return this.selectYesNo(container, this.profile.needsSponsorship === true);
        }

        // Relocation
        if (/relocat/i.test(text)) {
            return this.selectYesNo(container, true); // Default yes
        }

        // Background check
        if (/background.*check|screening/i.test(text)) {
            return this.selectYesNo(container, true); // Default yes
        }

        // Age verification
        if (/18.*years|at least 18/i.test(text)) {
            return this.selectYesNo(container, true);
        }

        return false;
    },

    /**
     * Select yes or no radio/checkbox in a container
     */
    selectYesNo(container, selectYes) {
        const inputs = container.querySelectorAll('input[type="radio"], input[type="checkbox"]');

        for (const input of inputs) {
            const label = document.querySelector(`label[for="${input.id}"]`) ||
                input.closest('label');
            const labelText = (label?.textContent || input.value || '').toLowerCase();

            const isYesOption = /^(yes|yeah|yep|i am|i do|correct)$/i.test(labelText.trim()) ||
                labelText.includes('yes');
            const isNoOption = /^(no|nah|nope|i am not|i do not)$/i.test(labelText.trim()) ||
                labelText.includes('no');

            if ((selectYes && isYesOption) || (!selectYes && isNoOption)) {
                input.checked = true;
                input.dispatchEvent(new Event('change', { bubbles: true }));
                return true;
            }
        }

        return false;
    }
};

// Export for content script usage
window.SmartAutofill = SmartAutofill;
