// exploit.team AutoJob API Client
// Handles all communication with job.exploit.team

const API_BASE = 'https://job.exploit.team/api';

class JobGenAPI {
    constructor() {
        this.token = null;
        this.user = null;
    }

    // Initialize from stored credentials
    async init() {
        const stored = await chrome.storage.local.get(['authToken', 'user']);
        if (stored.authToken) {
            this.token = stored.authToken;
            this.user = stored.user;
        }
        return this.isAuthenticated();
    }

    isAuthenticated() {
        return !!this.token;
    }

    getUser() {
        return this.user;
    }

    // Login with username and password
    async login(username, password) {
        try {
            const response = await fetch(`${API_BASE}/auth/login`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ username, password })
            });

            const data = await response.json();

            if (!response.ok || !data.success) {
                throw new Error(data.message || data.error || 'Login failed');
            }

            // API returns { success, data: { token, user } }
            this.token = data.data?.token || data.token;
            this.user = data.data?.user || data.user;

            // Store credentials
            await chrome.storage.local.set({
                authToken: this.token,
                user: this.user
            });

            // Auto-sync profile data from server
            await this.syncProfile();

            return { success: true, user: this.user };
        } catch (error) {
            console.error('Login error:', error);
            return { success: false, error: error.message };
        }
    }

    // Logout
    async logout() {
        this.token = null;
        this.user = null;
        await chrome.storage.local.remove(['authToken', 'user']);
        return { success: true };
    }

    // Generate tailored application materials
    async generate(jobData) {
        if (!this.isAuthenticated()) {
            throw new Error('Not authenticated');
        }

        try {
            // Combine additional info: location + user's saved additional info + job-specific comments
            const combinedAdditionalInfo = [
                jobData.location || '',
                jobData.additionalInfo || '',
                jobData.userComments ? `User notes for this job: ${jobData.userComments}` : ''
            ].filter(s => s.trim()).join('\n\n');

            const response = await fetch(`${API_BASE}/generate/simple`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                },
                body: JSON.stringify({
                    jobTitle: jobData.title,
                    company: jobData.company,
                    jobDescription: jobData.description,
                    additionalInfo: combinedAdditionalInfo,
                    masterResume: jobData.masterResume || ''
                })
            });

            const data = await response.json();

            if (!response.ok) {
                if (response.status === 401) {
                    await this.logout();
                    throw new Error('Session expired. Please login again.');
                }
                throw new Error(data.message || data.error || 'Generation failed');
            }

            return {
                success: true,
                resume: data.resume,
                coverLetter: data.coverLetter,
                emailDraft: data.emailDraft,
                generationId: data.generationId
            };
        } catch (error) {
            console.error('Generation error:', error);
            return { success: false, error: error.message };
        }
    }

    // Download document as PDF or DOCX
    async downloadDocument(type, content, format = 'pdf', style = 'professional-modern') {
        try {
            const response = await fetch(`${API_BASE}/generate/download`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                },
                body: JSON.stringify({
                    type,
                    content,
                    format,
                    style
                })
            });

            if (!response.ok) {
                const data = await response.json();
                throw new Error(data.message || 'Download failed');
            }

            // Return blob for download
            return await response.blob();
        } catch (error) {
            console.error('Download error:', error);
            throw error;
        }
    }

    // Get user's master resume and additional info
    async getResume() {
        if (!this.isAuthenticated()) {
            throw new Error('Not authenticated');
        }

        try {
            const response = await fetch(`${API_BASE}/user/resume`, {
                method: 'GET',
                headers: {
                    'Authorization': `Bearer ${this.token}`
                }
            });

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.message || 'Failed to get resume');
            }

            return {
                success: true,
                masterResume: data.masterResume,
                additionalInfo: data.additionalInfo
            };
        } catch (error) {
            console.error('Get resume error:', error);
            return { success: false, error: error.message };
        }
    }

    // Sync profile data from server to extension storage
    async syncProfile() {
        try {
            const result = await this.getResume();
            if (result.success) {
                await chrome.storage.local.set({
                    masterResume: result.masterResume || '',
                    additionalInfo: result.additionalInfo || ''
                });
                console.log('Profile synced from server');
                return { success: true };
            }
            return result;
        } catch (error) {
            console.error('Profile sync error:', error);
            return { success: false, error: error.message };
        }
    }

    // Generate answer for custom question
    async generateCustomAnswer(question, jobTitle, company, jobDescription) {
        if (!this.isAuthenticated()) {
            throw new Error('Not authenticated');
        }

        try {
            const response = await fetch(`${API_BASE}/generate/simple`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                },
                body: JSON.stringify({
                    jobTitle: jobTitle || 'Position',
                    company: company || 'Company',
                    jobDescription: `Answer this application question: "${question}"\n\nJob Context: ${jobDescription || 'N/A'}`,
                    additionalInfo: `Generate ONLY a direct answer to this question. Be professional, concise (2-4 sentences for short-answer, 1-2 paragraphs for essay questions). The question is: ${question}`
                })
            });

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.message || data.error || 'Generation failed');
            }

            // The API returns coverLetter, but we'll use it as the answer
            // since it's typically the free-text response
            return {
                success: true,
                answer: data.coverLetter || data.emailDraft || 'Generated answer not available'
            };
        } catch (error) {
            console.error('Custom answer generation error:', error);
            return { success: false, error: error.message };
        }
    }
}

// Export singleton instance
const api = new JobGenAPI();
export default api;
