// Indeed Auto-Apply Module
// Comprehensive automation for Indeed job applications

const IndeedAutoApply = {

    // Indeed Application Structure:
    // - Direct apply or redirect to company site
    // - Multi-step wizard for Indeed Apply
    // - Uses data-testid attributes extensively

    selectors: {
        // Apply button triggers
        applyButton: [
            '#indeedApplyButton',
            '[data-testid="indeedApply-button"]',
            '.jobsearch-IndeedApplyButton-contentWrapper',
            'button[id*="apply"]',
            '.ia-IndeedApplyButton'
        ],

        // Modal/frame container
        modal: [
            '#indeedApplyModal',
            '.indeed-apply-popup',
            'iframe[title*="Apply"]',
            '[data-testid="ia-container"]'
        ],

        // Form fields
        fields: {
            // Contact information
            firstName: ['input[name="firstName"]', 'input[id*="firstName"]', 'input[aria-label*="First"]'],
            lastName: ['input[name="lastName"]', 'input[id*="lastName"]', 'input[aria-label*="Last"]'],
            fullName: ['input[name="name"]', 'input[name="fullName"]', 'input[aria-label*="name"]'],
            email: ['input[name="email"]', 'input[type="email"]', 'input[id*="email"]'],
            phone: ['input[name="phone"]', 'input[type="tel"]', 'input[id*="phone"]'],

            // Address
            address: ['input[name="address"]', 'input[id*="address"]', 'input[aria-label*="Address"]'],
            city: ['input[name="city"]', 'input[id*="city"]'],
            state: ['select[name="state"]', 'input[name="state"]', 'select[id*="state"]'],
            zip: ['input[name="zip"]', 'input[name="postalCode"]', 'input[id*="zip"]'],
            country: ['select[name="country"]', 'select[id*="country"]'],

            // Resume
            resumeUpload: ['input[type="file"][name*="resume"]', 'input[type="file"][accept*="pdf"]'],

            // Cover letter
            coverLetter: ['textarea[name="coverLetter"]', 'textarea[id*="cover"]', 'textarea[aria-label*="cover"]'],
            message: ['textarea[name="message"]', 'textarea[id*="message"]', 'textarea[aria-label*="message"]'],

            // Work history
            currentJobTitle: ['input[name="currentJobTitle"]', 'input[id*="jobTitle"]'],
            currentCompany: ['input[name="currentCompany"]', 'input[id*="company"]'],

            // Experience
            yearsExperience: ['input[name="experience"]', 'select[name="experience"]', 'input[aria-label*="experience"]'],

            // Education
            degree: ['select[name="degree"]', 'input[name="degree"]'],
            school: ['input[name="school"]', 'input[name="university"]'],
            major: ['input[name="major"]', 'input[name="fieldOfStudy"]'],
            graduationYear: ['input[name="graduationYear"]', 'select[name="graduationYear"]'],

            // Salary
            salary: ['input[name="salary"]', 'input[name="desiredSalary"]', 'input[aria-label*="salary"]'],

            // Skills
            skills: ['input[name="skills"]', 'textarea[name="skills"]'],

            // URLs
            linkedin: ['input[name="linkedin"]', 'input[aria-label*="LinkedIn"]'],
            portfolio: ['input[name="portfolio"]', 'input[name="website"]', 'input[aria-label*="portfolio"]']
        },

        // Navigation
        nextButton: [
            'button[data-testid="ia-continueButton"]',
            'button[type="submit"]',
            '.ia-continueButton',
            'button:contains("Continue")',
            'button:contains("Next")'
        ],

        submitButton: [
            'button[data-testid="ia-submitButton"]',
            'button:contains("Submit")',
            'button:contains("Apply")',
            '.ia-submitButton'
        ],

        // Error messages
        errorMessage: [
            '.ia-errorMessage',
            '[data-testid="error-message"]',
            '.error-message',
            '[role="alert"]'
        ],

        // Yes/No radio buttons
        yesOption: ['input[value="yes"]', 'input[value="Yes"]', 'input[value="true"]', 'input[value="1"]'],
        noOption: ['input[value="no"]', 'input[value="No"]', 'input[value="false"]', 'input[value="0"]']
    },

    userProfile: null,

    async init(profile) {
        this.userProfile = profile || await this.loadUserProfile();
        return this;
    },

    async loadUserProfile() {
        const response = await chrome.runtime.sendMessage({ action: 'getUserProfile' });
        return response.success ? response.profile : null;
    },

    // Click apply button
    async clickApply() {
        for (const selector of this.selectors.applyButton) {
            const btn = document.querySelector(selector);
            if (btn && btn.offsetParent !== null) {
                btn.click();
                await this.wait(2000);
                return true;
            }
        }
        return false;
    },

    // Check if Indeed iframe is present
    getApplyFrame() {
        const iframe = document.querySelector('iframe[title*="Apply"], iframe[id*="indeed"]');
        return iframe?.contentDocument || document;
    },

    // Fill a field
    fillField(selectors, value, doc = document) {
        if (!value) return false;

        for (const selector of selectors) {
            try {
                const field = doc.querySelector(selector);
                if (field && field.offsetParent !== null) {
                    if (field.tagName === 'SELECT') {
                        return this.selectOption(field, value);
                    } else if (field.type === 'file') {
                        continue; // Handle separately
                    } else {
                        // Clear and set value
                        field.focus();
                        field.value = '';
                        field.value = value;
                        field.dispatchEvent(new Event('input', { bubbles: true }));
                        field.dispatchEvent(new Event('change', { bubbles: true }));
                        field.dispatchEvent(new Event('blur', { bubbles: true }));
                        return true;
                    }
                }
            } catch (e) {
                continue;
            }
        }
        return false;
    },

    // Select dropdown option
    selectOption(select, value) {
        const options = Array.from(select.options);

        // Exact match
        let option = options.find(o =>
            o.value.toLowerCase() === value.toLowerCase() ||
            o.text.toLowerCase() === value.toLowerCase()
        );

        // Partial match
        if (!option) {
            option = options.find(o =>
                o.text.toLowerCase().includes(value.toLowerCase()) ||
                o.value.toLowerCase().includes(value.toLowerCase())
            );
        }

        if (option) {
            select.value = option.value;
            select.dispatchEvent(new Event('change', { bubbles: true }));
            return true;
        }
        return false;
    },

    // Answer radio button questions
    answerQuestion(container, answer, doc = document) {
        const selectors = answer ? this.selectors.yesOption : this.selectors.noOption;

        for (const selector of selectors) {
            const radio = container.querySelector(selector);
            if (radio) {
                radio.click();
                return true;
            }
        }

        // Try by label
        const labels = container.querySelectorAll('label');
        const targetText = answer ? 'yes' : 'no';
        for (const label of labels) {
            if (label.textContent.toLowerCase().includes(targetText)) {
                label.click();
                return true;
            }
        }
        return false;
    },

    // Fill all visible fields
    async fillCurrentStep() {
        const profile = this.userProfile;
        if (!profile) return { filled: 0 };

        const doc = this.getApplyFrame();
        let filled = 0;

        // Name (some forms use full name, others split)
        if (profile.firstName && profile.lastName) {
            if (this.fillField(this.selectors.fields.firstName, profile.firstName, doc)) filled++;
            if (this.fillField(this.selectors.fields.lastName, profile.lastName, doc)) filled++;
            // Also try full name
            this.fillField(this.selectors.fields.fullName, `${profile.firstName} ${profile.lastName}`, doc);
        }

        // Contact
        if (this.fillField(this.selectors.fields.email, profile.email, doc)) filled++;
        if (this.fillField(this.selectors.fields.phone, profile.phone, doc)) filled++;

        // Address
        if (this.fillField(this.selectors.fields.address, profile.address, doc)) filled++;
        if (this.fillField(this.selectors.fields.city, profile.city, doc)) filled++;
        if (this.fillField(this.selectors.fields.state, profile.state, doc)) filled++;
        if (this.fillField(this.selectors.fields.zip, profile.zip, doc)) filled++;
        if (this.fillField(this.selectors.fields.country, profile.country || 'United States', doc)) filled++;

        // Current employment
        if (this.fillField(this.selectors.fields.currentJobTitle, profile.currentJobTitle, doc)) filled++;
        if (this.fillField(this.selectors.fields.currentCompany, profile.currentCompany, doc)) filled++;

        // Experience
        if (this.fillField(this.selectors.fields.yearsExperience, profile.yearsExperience?.toString(), doc)) filled++;

        // Education
        if (this.fillField(this.selectors.fields.degree, profile.degree, doc)) filled++;
        if (this.fillField(this.selectors.fields.school, profile.school, doc)) filled++;
        if (this.fillField(this.selectors.fields.major, profile.major, doc)) filled++;
        if (this.fillField(this.selectors.fields.graduationYear, profile.graduationYear?.toString(), doc)) filled++;

        // Salary
        if (profile.desiredSalary) {
            this.fillField(this.selectors.fields.salary, profile.desiredSalary.toString(), doc);
        }

        // URLs
        if (this.fillField(this.selectors.fields.linkedin, profile.linkedinUrl, doc)) filled++;
        if (this.fillField(this.selectors.fields.portfolio, profile.portfolioUrl, doc)) filled++;

        // Cover letter / Message
        if (profile.coverLetter) {
            if (this.fillField(this.selectors.fields.coverLetter, profile.coverLetter, doc)) filled++;
            this.fillField(this.selectors.fields.message, profile.coverLetter, doc);
        }

        // Skills
        if (profile.skills) {
            const skillsText = Array.isArray(profile.skills) ? profile.skills.join(', ') : profile.skills;
            this.fillField(this.selectors.fields.skills, skillsText, doc);
        }

        // Handle screening questions
        await this.handleScreeningQuestions(doc);

        return { filled };
    },

    // Handle common screening questions
    async handleScreeningQuestions(doc = document) {
        const profile = this.userProfile;
        const questions = doc.querySelectorAll('.ia-Question, [data-testid*="question"], .question-container, fieldset');

        for (const q of questions) {
            const text = q.textContent.toLowerCase();

            // Work authorization
            if (text.includes('authorized to work') || text.includes('legally authorized') || text.includes('eligible to work')) {
                this.answerQuestion(q, profile.workAuthorized !== false);
            }

            // Sponsorship
            else if (text.includes('sponsorship') || text.includes('require visa')) {
                this.answerQuestion(q, profile.needsSponsorship === true);
            }

            // Relocation
            else if (text.includes('willing to relocate') || text.includes('relocation')) {
                this.answerQuestion(q, profile.willingToRelocate === true);
            }

            // Years of experience (numeric input)
            else if (text.includes('years of experience') || text.includes('years experience')) {
                const input = q.querySelector('input[type="text"], input[type="number"]');
                if (input && profile.yearsExperience) {
                    input.value = profile.yearsExperience.toString();
                    input.dispatchEvent(new Event('input', { bubbles: true }));
                }
            }

            // Driver's license
            else if (text.includes("driver's license") || text.includes('valid driver')) {
                this.answerQuestion(q, profile.hasDriversLicense !== false);
            }

            // Background check
            else if (text.includes('background check')) {
                this.answerQuestion(q, true);
            }

            // Age requirement
            else if (text.includes('18 years') || text.includes('at least 18')) {
                this.answerQuestion(q, true);
            }

            // High school diploma
            else if (text.includes('high school') || text.includes('diploma') || text.includes('ged')) {
                this.answerQuestion(q, true);
            }

            // Physical requirements
            else if (text.includes('lift') || text.includes('physical') || text.includes('stand for')) {
                this.answerQuestion(q, true);
            }

            // Shift/schedule
            else if (text.includes('shift') || text.includes('weekend') || text.includes('overtime')) {
                this.answerQuestion(q, profile.willingToWorkWeekends !== false);
            }

            // Remote work
            else if (text.includes('remote') || text.includes('work from home')) {
                this.answerQuestion(q, true);
            }
        }
    },

    // Click next/continue button
    async clickNext() {
        const doc = this.getApplyFrame();

        for (const selector of this.selectors.nextButton) {
            try {
                const btn = doc.querySelector(selector);
                if (btn && !btn.disabled && btn.offsetParent !== null) {
                    btn.click();
                    await this.wait(1500);
                    return true;
                }
            } catch (e) {
                continue;
            }
        }
        return false;
    },

    // Check for submit button (final step)
    isSubmitStep() {
        const doc = this.getApplyFrame();
        for (const selector of this.selectors.submitButton) {
            if (doc.querySelector(selector)) return true;
        }
        return false;
    },

    // Check for errors
    hasErrors() {
        const doc = this.getApplyFrame();
        for (const selector of this.selectors.errorMessage) {
            const error = doc.querySelector(selector);
            if (error && error.textContent.trim()) {
                return error.textContent.trim();
            }
        }
        return false;
    },

    // Run complete auto-apply workflow
    async autoApply(options = { autoSubmit: false }) {
        const results = {
            success: false,
            stepsCompleted: 0,
            fieldsFilledTotal: 0,
            errors: [],
            message: ''
        };

        try {
            // Click apply
            if (!await this.clickApply()) {
                results.errors.push('Could not find Apply button');
                return results;
            }

            // Wait for form
            await this.wait(2000);

            // Process steps
            let maxSteps = 15;
            while (maxSteps-- > 0) {
                const { filled } = await this.fillCurrentStep();
                results.fieldsFilledTotal += filled;
                results.stepsCompleted++;

                const error = this.hasErrors();
                if (error) {
                    results.errors.push(error);
                }

                if (this.isSubmitStep()) {
                    if (options.autoSubmit) {
                        const doc = this.getApplyFrame();
                        for (const selector of this.selectors.submitButton) {
                            const btn = doc.querySelector(selector);
                            if (btn) {
                                btn.click();
                                await this.wait(2000);
                                results.success = true;
                                results.message = 'Application submitted!';
                                break;
                            }
                        }
                    } else {
                        results.success = true;
                        results.message = 'Application ready for review. Please verify and submit.';
                    }
                    break;
                }

                if (!await this.clickNext()) {
                    results.message = 'Completed all available steps.';
                    break;
                }

                await this.wait(1000);
            }

        } catch (error) {
            results.errors.push(error.message);
        }

        return results;
    },

    wait(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
};

window.IndeedAutoApply = IndeedAutoApply;
export default IndeedAutoApply;
