// LinkedIn Auto-Apply Module
// Comprehensive automation for LinkedIn Easy Apply workflow

const LinkedInAutoApply = {

    // LinkedIn Easy Apply Modal Structure:
    // - Multi-step wizard with "Next" buttons
    // - Contact info, resume upload, additional questions, review
    // - Uses aria-labels and data-test-* attributes

    selectors: {
        // Apply button triggers
        applyButton: [
            '.jobs-apply-button--top-card',
            '[data-control-name="jobdetails_topcard_inapply"]',
            'button[aria-label*="Easy Apply"]',
            '.jobs-s-apply button'
        ],

        // Modal container
        modal: [
            '.jobs-easy-apply-modal',
            '[data-test-modal-id="easy-apply-modal"]',
            '.artdeco-modal--layer-default'
        ],

        // Form fields
        fields: {
            // Contact information
            firstName: ['input[name="firstName"]', 'input[id*="first-name"]', 'input[aria-label*="First name"]'],
            lastName: ['input[name="lastName"]', 'input[id*="last-name"]', 'input[aria-label*="Last name"]'],
            email: ['input[name="email"]', 'input[type="email"]', 'input[aria-label*="Email"]'],
            phone: ['input[name="phone"]', 'input[type="tel"]', 'input[aria-label*="Phone"]', 'input[aria-label*="Mobile"]'],

            // Address
            city: ['input[aria-label*="City"]', 'input[name="city"]'],
            state: ['select[aria-label*="State"]', 'input[aria-label*="State"]'],
            zip: ['input[aria-label*="ZIP"]', 'input[aria-label*="Postal"]'],
            country: ['select[aria-label*="Country"]'],

            // Work authorization
            workAuth: ['select[aria-label*="authorized"]', 'select[aria-label*="work in"]'],
            sponsorship: ['select[aria-label*="sponsorship"]', 'select[aria-label*="visa"]'],

            // Experience
            yearsExperience: ['input[aria-label*="years of experience"]', 'select[aria-label*="experience"]'],

            // Salary
            salary: ['input[aria-label*="salary"]', 'input[aria-label*="compensation"]'],

            // Start date
            startDate: ['input[aria-label*="start"]', 'select[aria-label*="start"]'],

            // LinkedIn profile
            linkedin: ['input[aria-label*="LinkedIn"]', 'input[name="linkedin"]'],
            website: ['input[aria-label*="website"]', 'input[aria-label*="portfolio"]'],

            // Cover letter / Additional info
            coverLetter: ['textarea[aria-label*="cover"]', 'textarea[aria-label*="additional"]', 'textarea[name="coverLetter"]'],

            // Resume upload
            resumeUpload: ['input[type="file"]', 'input[accept*="pdf"]', 'input[accept*="doc"]']
        },

        // Navigation buttons
        nextButton: ['button[aria-label="Continue to next step"]', 'button[aria-label="Next"]', 'button[data-control-name="continue_unify"]'],
        submitButton: ['button[aria-label="Submit application"]', 'button[aria-label="Review your application"]', 'button[data-control-name="submit_unify"]'],
        reviewButton: ['button[aria-label="Review"]', 'button[data-easy-apply-next-button]'],

        // Error/validation
        errorMessage: ['.artdeco-inline-feedback--error', '.fb-form-element__error'],

        // Radio buttons and checkboxes (yes/no questions)
        yesRadio: ['input[type="radio"][value="Yes"]', 'label[for*="yes"]', 'input[value="true"]'],
        noRadio: ['input[type="radio"][value="No"]', 'label[for*="no"]', 'input[value="false"]']
    },

    // User profile data (populated from storage)
    userProfile: null,

    // Initialize with user data
    async init(profile) {
        this.userProfile = profile || await this.loadUserProfile();
        return this;
    },

    async loadUserProfile() {
        const response = await chrome.runtime.sendMessage({ action: 'getUserProfile' });
        return response.success ? response.profile : null;
    },

    // Click the Easy Apply button
    async clickApply() {
        for (const selector of this.selectors.applyButton) {
            const btn = document.querySelector(selector);
            if (btn) {
                btn.click();
                await this.wait(1500);
                return true;
            }
        }
        return false;
    },

    // Wait for modal to appear
    async waitForModal(timeout = 5000) {
        const start = Date.now();
        while (Date.now() - start < timeout) {
            for (const selector of this.selectors.modal) {
                if (document.querySelector(selector)) {
                    await this.wait(500);
                    return true;
                }
            }
            await this.wait(200);
        }
        return false;
    },

    // Fill a single field
    fillField(selectors, value) {
        if (!value) return false;

        for (const selector of selectors) {
            const field = document.querySelector(selector);
            if (field && field.offsetParent !== null) { // Check if visible
                if (field.tagName === 'SELECT') {
                    // Handle dropdown
                    this.selectOption(field, value);
                } else if (field.type === 'file') {
                    // Skip file inputs here (handled separately)
                    continue;
                } else {
                    // Handle text input
                    field.value = value;
                    field.dispatchEvent(new Event('input', { bubbles: true }));
                    field.dispatchEvent(new Event('change', { bubbles: true }));
                    field.dispatchEvent(new Event('blur', { bubbles: true }));
                }
                return true;
            }
        }
        return false;
    },

    // Select dropdown option
    selectOption(select, value) {
        // Try exact match first
        for (const option of select.options) {
            if (option.value.toLowerCase() === value.toLowerCase() ||
                option.text.toLowerCase() === value.toLowerCase()) {
                select.value = option.value;
                select.dispatchEvent(new Event('change', { bubbles: true }));
                return true;
            }
        }

        // Try partial match
        for (const option of select.options) {
            if (option.text.toLowerCase().includes(value.toLowerCase())) {
                select.value = option.value;
                select.dispatchEvent(new Event('change', { bubbles: true }));
                return true;
            }
        }

        // Select first non-empty option as fallback
        if (select.options.length > 1) {
            select.selectedIndex = 1;
            select.dispatchEvent(new Event('change', { bubbles: true }));
        }
        return false;
    },

    // Answer yes/no questions
    answerYesNo(container, answer = 'yes') {
        const selectors = answer === 'yes' ? this.selectors.yesRadio : this.selectors.noRadio;

        for (const selector of selectors) {
            const radio = container.querySelector(selector);
            if (radio) {
                radio.click();
                return true;
            }
        }

        // Try by label text
        const labels = container.querySelectorAll('label');
        for (const label of labels) {
            const text = label.textContent.toLowerCase();
            if ((answer === 'yes' && text.includes('yes')) ||
                (answer === 'no' && text.includes('no'))) {
                label.click();
                return true;
            }
        }
        return false;
    },

    // Fill all visible form fields on current step
    async fillCurrentStep() {
        const profile = this.userProfile;
        if (!profile) return { filled: 0, total: 0 };

        let filled = 0;
        let total = 0;

        // Contact info
        if (this.fillField(this.selectors.fields.firstName, profile.firstName)) filled++;
        if (this.fillField(this.selectors.fields.lastName, profile.lastName)) filled++;
        if (this.fillField(this.selectors.fields.email, profile.email)) filled++;
        if (this.fillField(this.selectors.fields.phone, profile.phone)) filled++;

        // Address
        if (this.fillField(this.selectors.fields.city, profile.city)) filled++;
        if (this.fillField(this.selectors.fields.state, profile.state)) filled++;
        if (this.fillField(this.selectors.fields.zip, profile.zip)) filled++;
        if (this.fillField(this.selectors.fields.country, profile.country || 'United States')) filled++;

        // Work authorization (typically "Yes" for US citizens/residents)
        this.fillField(this.selectors.fields.workAuth, profile.workAuthorized ? 'Yes' : 'No');
        this.fillField(this.selectors.fields.sponsorship, profile.needsSponsorship ? 'Yes' : 'No');

        // Experience
        if (this.fillField(this.selectors.fields.yearsExperience, profile.yearsExperience?.toString())) filled++;

        // Salary
        if (profile.desiredSalary) {
            this.fillField(this.selectors.fields.salary, profile.desiredSalary.toString());
        }

        // URLs
        if (this.fillField(this.selectors.fields.linkedin, profile.linkedinUrl)) filled++;
        if (this.fillField(this.selectors.fields.website, profile.portfolioUrl)) filled++;

        // Cover letter (if generated)
        if (profile.coverLetter) {
            if (this.fillField(this.selectors.fields.coverLetter, profile.coverLetter)) filled++;
        }

        // Fill any visible additional textareas
        const textareas = document.querySelectorAll('.jobs-easy-apply-modal textarea:not([disabled])');
        for (const ta of textareas) {
            if (!ta.value && profile.coverLetter) {
                total++;
                ta.value = profile.coverLetter;
                ta.dispatchEvent(new Event('input', { bubbles: true }));
                filled++;
            }
        }

        // Handle yes/no questions intelligently
        await this.handleQuestions();

        return { filled, total };
    },

    // Handle common screening questions
    async handleQuestions() {
        const profile = this.userProfile;
        const modal = document.querySelector('.jobs-easy-apply-modal, .artdeco-modal');
        if (!modal) return;

        // Find all question containers
        const questions = modal.querySelectorAll('.jobs-easy-apply-form-section__grouping, .fb-dash-form-element');

        for (const q of questions) {
            const text = q.textContent.toLowerCase();

            // Work authorization
            if (text.includes('authorized to work') || text.includes('legally authorized')) {
                this.answerYesNo(q, profile.workAuthorized !== false ? 'yes' : 'no');
            }

            // Sponsorship
            else if (text.includes('sponsorship') || text.includes('visa')) {
                this.answerYesNo(q, profile.needsSponsorship ? 'yes' : 'no');
            }

            // Relocation
            else if (text.includes('relocate') || text.includes('relocation')) {
                this.answerYesNo(q, profile.willingToRelocate ? 'yes' : 'no');
            }

            // Background check
            else if (text.includes('background check')) {
                this.answerYesNo(q, 'yes');
            }

            // Drug test
            else if (text.includes('drug test') || text.includes('drug screening')) {
                this.answerYesNo(q, 'yes');
            }

            // Non-compete / NDA
            else if (text.includes('non-compete') || text.includes('nda') || text.includes('confidentiality')) {
                this.answerYesNo(q, 'yes');
            }

            // 18+ years old
            else if (text.includes('18 years') || text.includes('over 18')) {
                this.answerYesNo(q, 'yes');
            }

            // Disability / veteran status - prefer not to disclose
            else if (text.includes('disability') || text.includes('veteran')) {
                const prefNotDisclose = q.querySelector('input[value*="prefer"], label:contains("prefer")');
                if (prefNotDisclose) prefNotDisclose.click();
            }
        }
    },

    // Click next/continue button
    async clickNext() {
        for (const selector of this.selectors.nextButton) {
            const btn = document.querySelector(selector);
            if (btn && !btn.disabled && btn.offsetParent !== null) {
                btn.click();
                await this.wait(1000);
                return true;
            }
        }
        return false;
    },

    // Click review button
    async clickReview() {
        for (const selector of this.selectors.reviewButton) {
            const btn = document.querySelector(selector);
            if (btn && !btn.disabled) {
                btn.click();
                await this.wait(1000);
                return true;
            }
        }
        return false;
    },

    // Check if there are validation errors
    hasErrors() {
        for (const selector of this.selectors.errorMessage) {
            const error = document.querySelector(selector);
            if (error && error.textContent.trim()) {
                return error.textContent.trim();
            }
        }
        return false;
    },

    // Check if we're on the final review/submit step
    isReviewStep() {
        return document.querySelector('button[aria-label="Submit application"]') !== null ||
            document.querySelector('[data-control-name="submit_unify"]') !== null;
    },

    // Run the complete auto-apply workflow
    async autoApply(options = { autoSubmit: false }) {
        const results = {
            success: false,
            stepsCompleted: 0,
            fieldsFilledTotal: 0,
            errors: [],
            message: ''
        };

        try {
            // Step 1: Click Apply button
            if (!await this.clickApply()) {
                results.errors.push('Could not find Easy Apply button');
                return results;
            }

            // Step 2: Wait for modal
            if (!await this.waitForModal()) {
                results.errors.push('Application modal did not appear');
                return results;
            }

            // Step 3: Process each step
            let maxSteps = 10; // Safety limit
            while (maxSteps-- > 0) {
                // Fill current step
                const { filled } = await this.fillCurrentStep();
                results.fieldsFilledTotal += filled;

                // Check for errors
                const error = this.hasErrors();
                if (error) {
                    results.errors.push(`Validation error: ${error}`);
                }

                // Check if review step
                if (this.isReviewStep()) {
                    results.stepsCompleted++;
                    if (options.autoSubmit) {
                        // Submit application
                        const submitBtn = document.querySelector('button[aria-label="Submit application"]');
                        if (submitBtn) {
                            submitBtn.click();
                            await this.wait(2000);
                            results.success = true;
                            results.message = 'Application submitted successfully!';
                        }
                    } else {
                        results.success = true;
                        results.message = 'Application ready for review. Please verify and submit.';
                    }
                    break;
                }

                // Click next
                if (!await this.clickNext()) {
                    // Try review button
                    if (!await this.clickReview()) {
                        results.message = 'Completed all fillable fields. Manual navigation may be needed.';
                        break;
                    }
                }

                results.stepsCompleted++;
                await this.wait(800);
            }

        } catch (error) {
            results.errors.push(error.message);
        }

        return results;
    },

    // Utility: wait
    wait(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
};

// Export for use in content script
window.LinkedInAutoApply = LinkedInAutoApply;
export default LinkedInAutoApply;
