// AutoApply Orchestrator
// Coordinates the complete auto-apply workflow

import UserProfile from './profile.js';
import ResumeUploader from './uploader.js';

const AutoApplyOrchestrator = {

    // Current state
    state: {
        jobData: null,
        userProfile: null,
        generatedContent: null,
        site: null
    },

    // Initialize with job data
    async init(jobData) {
        this.state.jobData = jobData;
        this.state.site = this.detectSite();
        this.state.userProfile = await UserProfile.load();
        return this;
    },

    // Detect current site
    detectSite() {
        const hostname = window.location.hostname;
        if (hostname.includes('linkedin.com')) return 'linkedin';
        if (hostname.includes('indeed.com')) return 'indeed';
        if (hostname.includes('greenhouse.io')) return 'greenhouse';
        if (hostname.includes('lever.co')) return 'lever';
        if (hostname.includes('workday')) return 'workday';
        return 'generic';
    },

    // Validate readiness
    async validate() {
        const issues = [];

        // Check user profile
        const profileValidation = await UserProfile.validateForAutoApply();
        if (!profileValidation.valid) {
            issues.push(`Missing profile fields: ${profileValidation.missing.join(', ')}`);
        }

        // Check job data
        if (!this.state.jobData?.title) {
            issues.push('Job data not extracted');
        }

        return {
            ready: issues.length === 0,
            issues
        };
    },

    // Generate tailored content using API
    async generateContent() {
        const { jobData, userProfile } = this.state;

        try {
            // Call the API to generate content
            const response = await chrome.runtime.sendMessage({
                action: 'generate',
                data: {
                    title: jobData.title,
                    company: jobData.company,
                    description: jobData.description,
                    location: jobData.location,
                    jobUrl: jobData.jobUrl
                }
            });

            if (!response.success) {
                throw new Error(response.error || 'Generation failed');
            }

            this.state.generatedContent = {
                resume: response.resume,
                coverLetter: response.coverLetter,
                emailDraft: response.emailDraft
            };

            // Store in profile for auto-fill
            await UserProfile.setGeneratedContent(
                response.resume,
                response.coverLetter,
                response.emailDraft
            );

            // Update profile with cover letter for this application
            this.state.userProfile.coverLetter = response.coverLetter;

            return { success: true, content: this.state.generatedContent };

        } catch (error) {
            return { success: false, error: error.message };
        }
    },

    // Run the complete workflow
    async run(options = { autoSubmit: false, uploadResume: true }) {
        const results = {
            success: false,
            steps: [],
            errors: [],
            finalMessage: ''
        };

        try {
            // Step 1: Validate
            results.steps.push({ name: 'Validate', status: 'running' });
            const validation = await this.validate();
            if (!validation.ready) {
                results.steps[0].status = 'warning';
                results.steps[0].message = validation.issues.join('; ');
                // Continue anyway with partial data
            } else {
                results.steps[0].status = 'complete';
            }

            // Step 2: Generate content
            results.steps.push({ name: 'Generate tailored resume', status: 'running' });
            const genResult = await this.generateContent();
            if (!genResult.success) {
                results.steps[1].status = 'error';
                results.steps[1].message = genResult.error;
                results.errors.push(`Content generation failed: ${genResult.error}`);
                // Can still try to apply with existing profile data
            } else {
                results.steps[1].status = 'complete';
            }

            // Step 3: Click apply and wait for form
            results.steps.push({ name: 'Open application form', status: 'running' });
            const applyClicked = await this.clickApplyButton();
            if (!applyClicked) {
                results.steps[2].status = 'error';
                results.errors.push('Could not find or click Apply button');
                return results;
            }
            results.steps[2].status = 'complete';

            await this.wait(2000);

            // Step 4: Fill form
            results.steps.push({ name: 'Fill application form', status: 'running' });
            const fillResult = await this.fillForm();
            results.steps[3].status = 'complete';
            results.steps[3].message = `Filled ${fillResult.fieldsFilledTotal} fields across ${fillResult.stepsCompleted} steps`;

            // Step 5: Upload resume if needed
            if (options.uploadResume && this.state.generatedContent?.resume) {
                results.steps.push({ name: 'Upload resume', status: 'running' });
                const uploadResult = await ResumeUploader.downloadAndUpload(
                    'resume',
                    this.state.generatedContent.resume,
                    'pdf',
                    'professional-modern'
                );
                if (uploadResult.success) {
                    results.steps[4].status = 'complete';
                    results.steps[4].message = `Uploaded ${uploadResult.filename}`;
                } else {
                    results.steps[4].status = 'warning';
                    results.steps[4].message = 'Resume upload may require manual action';
                }
            }

            // Step 6: Review or submit
            if (options.autoSubmit) {
                results.steps.push({ name: 'Submit application', status: 'running' });
                const submitted = await this.submitApplication();
                results.steps[results.steps.length - 1].status = submitted ? 'complete' : 'error';
                results.success = submitted;
                results.finalMessage = submitted ?
                    'Application submitted successfully!' :
                    'Ready for review - please verify and submit manually';
            } else {
                results.success = true;
                results.finalMessage = 'Application form filled. Please review and submit.';
            }

            // Log application
            if (results.success) {
                await chrome.runtime.sendMessage({
                    action: 'logApplication',
                    data: {
                        site: this.state.site,
                        title: this.state.jobData.title,
                        company: this.state.jobData.company,
                        jobUrl: this.state.jobData.jobUrl
                    }
                });
            }

        } catch (error) {
            results.errors.push(error.message);
            results.finalMessage = `Error: ${error.message}`;
        }

        return results;
    },

    // Click the apply button for current site
    async clickApplyButton() {
        const selectors = {
            linkedin: ['.jobs-apply-button--top-card', '[data-control-name="jobdetails_topcard_inapply"]'],
            indeed: ['#indeedApplyButton', '[data-testid="indeedApply-button"]'],
            greenhouse: ['#apply_button', 'a[data-mapped="true"]'],
            lever: ['.postings-btn-wrapper a', '[data-qa="btn-apply"]'],
            workday: ['[data-automation-id="applyButton"]'],
            generic: ['button:contains("Apply")', 'a:contains("Apply")', '.apply-button']
        };

        const siteSelectors = selectors[this.state.site] || selectors.generic;

        for (const selector of siteSelectors) {
            try {
                const btn = document.querySelector(selector);
                if (btn && btn.offsetParent !== null) {
                    btn.click();
                    return true;
                }
            } catch (e) {
                // Try next selector
            }
        }

        // Try generic approach
        const allButtons = document.querySelectorAll('button, a.btn, a.button');
        for (const btn of allButtons) {
            const text = btn.textContent.toLowerCase();
            if ((text.includes('apply') || text.includes('easy apply')) && btn.offsetParent !== null) {
                btn.click();
                return true;
            }
        }

        return false;
    },

    // Fill form based on site
    async fillForm() {
        const profile = this.state.userProfile;
        let result = { stepsCompleted: 0, fieldsFilledTotal: 0 };

        // Common field mappings
        const fieldMappings = [
            { selectors: ['input[name="firstName"]', 'input[id*="firstName"]', 'input[aria-label*="First"]'], value: profile.firstName },
            { selectors: ['input[name="lastName"]', 'input[id*="lastName"]', 'input[aria-label*="Last"]'], value: profile.lastName },
            { selectors: ['input[name="email"]', 'input[type="email"]'], value: profile.email },
            { selectors: ['input[name="phone"]', 'input[type="tel"]'], value: profile.phone },
            { selectors: ['input[name="city"]', 'input[aria-label*="City"]'], value: profile.city },
            { selectors: ['input[name="state"]', 'select[name="state"]'], value: profile.state },
            { selectors: ['input[name="zip"]', 'input[name="postalCode"]'], value: profile.zip },
            { selectors: ['textarea[name="coverLetter"]', 'textarea[aria-label*="cover"]'], value: profile.coverLetter },
            { selectors: ['input[name="linkedin"]', 'input[aria-label*="LinkedIn"]'], value: profile.linkedinUrl },
            { selectors: ['input[name="website"]', 'input[name="portfolio"]'], value: profile.portfolioUrl }
        ];

        // Process each step of multi-step form
        let maxSteps = 10;
        while (maxSteps-- > 0) {
            let filledThisStep = 0;

            // Fill all visible fields
            for (const mapping of fieldMappings) {
                if (!mapping.value) continue;

                for (const selector of mapping.selectors) {
                    try {
                        const field = document.querySelector(selector);
                        if (field && field.offsetParent !== null && !field.value) {
                            if (field.tagName === 'SELECT') {
                                this.selectOption(field, mapping.value);
                            } else {
                                field.value = mapping.value;
                                field.dispatchEvent(new Event('input', { bubbles: true }));
                                field.dispatchEvent(new Event('change', { bubbles: true }));
                            }
                            filledThisStep++;
                            result.fieldsFilledTotal++;
                            break;
                        }
                    } catch (e) {
                        continue;
                    }
                }
            }

            // Handle radio buttons (yes/no questions)
            await this.handleYesNoQuestions(profile);

            result.stepsCompleted++;

            // Try to go to next step
            const nextClicked = await this.clickNextButton();
            if (!nextClicked) break;

            await this.wait(1000);
        }

        return result;
    },

    // Handle yes/no screening questions
    async handleYesNoQuestions(profile) {
        const questions = document.querySelectorAll('fieldset, .question, [role="group"]');

        for (const q of questions) {
            const text = q.textContent.toLowerCase();
            let answer = null;

            if (text.includes('authorized to work')) answer = profile.workAuthorized !== false;
            else if (text.includes('sponsorship')) answer = profile.needsSponsorship === true;
            else if (text.includes('relocate')) answer = profile.willingToRelocate === true;
            else if (text.includes('background check')) answer = true;
            else if (text.includes('18 years')) answer = true;

            if (answer !== null) {
                const radioValue = answer ? 'yes' : 'no';
                const radio = q.querySelector(`input[value="${radioValue}"], input[value="${radioValue.charAt(0).toUpperCase() + radioValue.slice(1)}"]`);
                if (radio) radio.click();
            }
        }
    },

    // Select dropdown option
    selectOption(select, value) {
        for (const option of select.options) {
            if (option.text.toLowerCase().includes(value.toLowerCase()) ||
                option.value.toLowerCase().includes(value.toLowerCase())) {
                select.value = option.value;
                select.dispatchEvent(new Event('change', { bubbles: true }));
                return true;
            }
        }
        return false;
    },

    // Click next/continue button
    async clickNextButton() {
        const nextSelectors = [
            'button[aria-label="Continue"]',
            'button[aria-label="Next"]',
            'button[type="submit"]:not([aria-label*="Submit"])',
            '.continue-button',
            '.next-button'
        ];

        for (const selector of nextSelectors) {
            const btn = document.querySelector(selector);
            if (btn && !btn.disabled && btn.offsetParent !== null) {
                btn.click();
                return true;
            }
        }

        // Generic approach
        const buttons = document.querySelectorAll('button');
        for (const btn of buttons) {
            const text = btn.textContent.toLowerCase();
            if ((text === 'next' || text === 'continue') && !btn.disabled) {
                btn.click();
                return true;
            }
        }

        return false;
    },

    // Submit the application
    async submitApplication() {
        const submitSelectors = [
            'button[aria-label="Submit application"]',
            'button[aria-label="Submit"]',
            'button[type="submit"]',
            '.submit-button'
        ];

        for (const selector of submitSelectors) {
            const btn = document.querySelector(selector);
            if (btn && !btn.disabled) {
                btn.click();
                await this.wait(2000);
                return true;
            }
        }
        return false;
    },

    wait(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
};

window.AutoApplyOrchestrator = AutoApplyOrchestrator;
export default AutoApplyOrchestrator;
