// User Profile Module
// Stores and manages user information for auto-fill

const UserProfile = {

    // Default profile structure
    defaultProfile: {
        // Personal
        firstName: '',
        lastName: '',
        email: '',
        phone: '',

        // Address
        address: '',
        city: '',
        state: '',
        zip: '',
        country: 'United States',

        // Employment
        currentJobTitle: '',
        currentCompany: '',
        yearsExperience: 0,
        desiredSalary: null,

        // Education
        degree: '',
        school: '',
        major: '',
        graduationYear: null,

        // Skills (array or comma-separated string)
        skills: [],

        // URLs
        linkedinUrl: '',
        portfolioUrl: '',
        githubUrl: '',

        // Work preferences
        workAuthorized: true,
        needsSponsorship: false,
        willingToRelocate: false,
        willingToWorkWeekends: true,
        hasDriversLicense: true,

        // Resume (stored as base64 or text)
        resumeText: '',
        resumeFileName: '',

        // Cover letter template
        coverLetterTemplate: '',

        // Generated content (from AI)
        generatedResume: '',
        generatedCoverLetter: '',
        generatedEmailDraft: ''
    },

    // Load profile from storage
    async load() {
        const result = await chrome.storage.local.get('userProfile');
        return { ...this.defaultProfile, ...(result.userProfile || {}) };
    },

    // Save profile to storage
    async save(profile) {
        const merged = { ...this.defaultProfile, ...profile };
        await chrome.storage.local.set({ userProfile: merged });
        return merged;
    },

    // Update specific fields
    async update(updates) {
        const current = await this.load();
        const updated = { ...current, ...updates };
        return await this.save(updated);
    },

    // Set generated content from AI
    async setGeneratedContent(resume, coverLetter, emailDraft) {
        return await this.update({
            generatedResume: resume,
            generatedCoverLetter: coverLetter,
            generatedEmailDraft: emailDraft
        });
    },

    // Get full name
    async getFullName() {
        const profile = await this.load();
        return `${profile.firstName} ${profile.lastName}`.trim();
    },

    // Validate required fields for auto-apply
    async validateForAutoApply() {
        const profile = await this.load();
        const missing = [];

        if (!profile.firstName) missing.push('First Name');
        if (!profile.lastName) missing.push('Last Name');
        if (!profile.email) missing.push('Email');
        if (!profile.phone) missing.push('Phone');

        return {
            valid: missing.length === 0,
            missing
        };
    },

    // Import from resume text (parse basic info)
    async importFromResume(resumeText) {
        // Simple parsing - can be enhanced
        const lines = resumeText.split('\n').filter(l => l.trim());
        const updates = {};

        // Try to find email
        const emailMatch = resumeText.match(/[\w.-]+@[\w.-]+\.\w+/);
        if (emailMatch) updates.email = emailMatch[0];

        // Try to find phone
        const phoneMatch = resumeText.match(/[\(]?\d{3}[\)]?[-.\s]?\d{3}[-.\s]?\d{4}/);
        if (phoneMatch) updates.phone = phoneMatch[0];

        // Try to find LinkedIn
        const linkedinMatch = resumeText.match(/linkedin\.com\/in\/[\w-]+/i);
        if (linkedinMatch) updates.linkedinUrl = `https://www.${linkedinMatch[0]}`;

        // Try to find GitHub
        const githubMatch = resumeText.match(/github\.com\/[\w-]+/i);
        if (githubMatch) updates.githubUrl = `https://${githubMatch[0]}`;

        // Store the resume text
        updates.resumeText = resumeText;

        if (Object.keys(updates).length > 0) {
            await this.update(updates);
        }

        return updates;
    },

    // Export profile as JSON
    async export() {
        const profile = await this.load();
        return JSON.stringify(profile, null, 2);
    },

    // Import profile from JSON
    async import(jsonString) {
        try {
            const parsed = JSON.parse(jsonString);
            return await this.save(parsed);
        } catch (e) {
            throw new Error('Invalid profile JSON');
        }
    },

    // Clear profile
    async clear() {
        await chrome.storage.local.remove('userProfile');
        return this.defaultProfile;
    }
};

export default UserProfile;
