// Resume Upload Helper
// Handles file creation and upload for auto-apply

const ResumeUploader = {

    // Create a File object from text content
    createFileFromText(content, filename, mimeType = 'application/pdf') {
        const blob = new Blob([content], { type: mimeType });
        return new File([blob], filename, { type: mimeType });
    },

    // Create File from base64 data
    createFileFromBase64(base64Data, filename, mimeType = 'application/pdf') {
        const byteString = atob(base64Data.split(',')[1] || base64Data);
        const ab = new ArrayBuffer(byteString.length);
        const ia = new Uint8Array(ab);

        for (let i = 0; i < byteString.length; i++) {
            ia[i] = byteString.charCodeAt(i);
        }

        const blob = new Blob([ab], { type: mimeType });
        return new File([blob], filename, { type: mimeType });
    },

    // Find file input on page
    findFileInput(doc = document) {
        const selectors = [
            'input[type="file"][accept*="pdf"]',
            'input[type="file"][accept*="doc"]',
            'input[type="file"][name*="resume"]',
            'input[type="file"][name*="cv"]',
            'input[type="file"][id*="resume"]',
            'input[type="file"]'
        ];

        for (const selector of selectors) {
            const input = doc.querySelector(selector);
            if (input) return input;
        }
        return null;
    },

    // Upload file to input
    async uploadToInput(fileInput, file) {
        if (!fileInput || !file) return false;

        // Create a DataTransfer object to set files
        const dataTransfer = new DataTransfer();
        dataTransfer.items.add(file);
        fileInput.files = dataTransfer.files;

        // Dispatch events
        fileInput.dispatchEvent(new Event('change', { bubbles: true }));
        fileInput.dispatchEvent(new Event('input', { bubbles: true }));

        return true;
    },

    // Download resume from API and upload
    async downloadAndUpload(type, content, format, style, doc = document) {
        try {
            // Request download from background
            const response = await chrome.runtime.sendMessage({
                action: 'download',
                data: { type, content, format, style }
            });

            if (!response.success) {
                throw new Error(response.error || 'Download failed');
            }

            // Create file from response
            const filename = response.filename || `${type}.${format}`;
            const mimeType = format === 'pdf' ? 'application/pdf' :
                'application/vnd.openxmlformats-officedocument.wordprocessingml.document';

            const file = this.createFileFromBase64(response.data, filename, mimeType);

            // Find and upload to input
            const fileInput = this.findFileInput(doc);
            if (fileInput) {
                await this.uploadToInput(fileInput, file);
                return { success: true, filename };
            } else {
                return { success: false, error: 'No file input found' };
            }

        } catch (error) {
            return { success: false, error: error.message };
        }
    },

    // Check if resume upload is required on current page
    isResumeUploadRequired(doc = document) {
        const fileInput = this.findFileInput(doc);
        if (!fileInput) return false;

        // Check if it's required
        if (fileInput.required) return true;

        // Check parent labels for "required" text
        const parent = fileInput.closest('.form-group, .field, fieldset, label');
        if (parent) {
            const text = parent.textContent.toLowerCase();
            if (text.includes('required') || text.includes('*')) return true;
        }

        return false;
    },

    // Check if file is already uploaded
    isFileUploaded(doc = document) {
        // Look for uploaded file indicators
        const indicators = [
            '.file-uploaded',
            '.upload-complete',
            '[data-testid="uploaded-file"]',
            '.resume-preview',
            '.file-name:not(:empty)'
        ];

        for (const selector of indicators) {
            if (doc.querySelector(selector)) return true;
        }

        // Check if file input has files
        const fileInput = this.findFileInput(doc);
        if (fileInput && fileInput.files.length > 0) return true;

        return false;
    }
};

export default ResumeUploader;
