// Generic Job Extractor
// Attempts to extract job information from any page using common patterns

const GenericExtractor = {
    name: 'Generic',

    // Generic pages are always considered potential job pages
    // The extraction quality will vary
    isJobPage() {
        // Look for common job-related keywords in the page
        const pageText = document.body.textContent.toLowerCase();
        const jobKeywords = ['apply now', 'job description', 'requirements', 'qualifications', 'responsibilities', 'we are looking for', 'about the role'];

        const hasJobContent = jobKeywords.some(keyword => pageText.includes(keyword));
        const hasJobTitle = this.findJobTitle() !== '';

        return hasJobContent && hasJobTitle;
    },

    // Find job title using common patterns
    findJobTitle() {
        // Try common job title locations
        const selectors = [
            'h1',
            '[class*="job-title"]',
            '[class*="jobTitle"]',
            '[class*="position-title"]',
            '[data-testid*="title"]',
            '.posting-headline h2',
            '[itemprop="title"]'
        ];

        for (const selector of selectors) {
            const elements = document.querySelectorAll(selector);
            for (const el of elements) {
                const text = el.textContent.trim();
                // Job titles are typically short and don't contain paragraphs
                if (text.length > 5 && text.length < 150 && !text.includes('\n\n')) {
                    return text;
                }
            }
        }

        // Fallback: check meta tags
        const ogTitle = document.querySelector('meta[property="og:title"]');
        if (ogTitle) {
            return ogTitle.content;
        }

        return '';
    },

    // Find company name using common patterns
    findCompany() {
        const selectors = [
            '[class*="company-name"]',
            '[class*="companyName"]',
            '[class*="employer"]',
            '[itemprop="hiringOrganization"]',
            '[data-testid*="company"]'
        ];

        for (const selector of selectors) {
            const el = document.querySelector(selector);
            if (el) {
                const text = el.textContent.trim();
                if (text.length > 1 && text.length < 100) {
                    return text;
                }
            }
        }

        // Try meta tags
        const author = document.querySelector('meta[name="author"]');
        if (author) {
            return author.content;
        }

        // Try to extract from page title
        const pageTitle = document.title;
        const patterns = [
            /at\s+(.+?)(?:\s*[-|]|$)/i,
            /(.+?)\s+[-|]\s+careers/i,
            /jobs?\s+at\s+(.+)/i
        ];

        for (const pattern of patterns) {
            const match = pageTitle.match(pattern);
            if (match) {
                return match[1].trim();
            }
        }

        return '';
    },

    // Find location
    findLocation() {
        const selectors = [
            '[class*="location"]',
            '[class*="Location"]',
            '[itemprop="jobLocation"]',
            '[data-testid*="location"]'
        ];

        for (const selector of selectors) {
            const el = document.querySelector(selector);
            if (el) {
                const text = el.textContent.trim();
                if (text.length > 2 && text.length < 100) {
                    return text;
                }
            }
        }

        // Look for location patterns in text
        const pageText = document.body.textContent;
        const locationPatterns = [
            /(?:location|located?\s+in|based\s+in)[:\s]+([^.]+)/i,
            /([A-Z][a-z]+(?:,\s*[A-Z]{2})?)\s*(?:•|—|-)\s*(?:remote|hybrid|on-?site)/i
        ];

        for (const pattern of locationPatterns) {
            const match = pageText.match(pattern);
            if (match) {
                return match[1].trim();
            }
        }

        return 'Location not specified';
    },

    // Find job description
    findDescription() {
        const selectors = [
            '[class*="job-description"]',
            '[class*="jobDescription"]',
            '[class*="description"]',
            '[itemprop="description"]',
            '#job-description',
            '.posting-description',
            'article'
        ];

        for (const selector of selectors) {
            const el = document.querySelector(selector);
            if (el) {
                const text = el.textContent.trim();
                if (text.length > 200) {
                    return text;
                }
            }
        }

        // Fallback: find the longest text block
        const paragraphs = document.querySelectorAll('p, li, div');
        let longestText = '';

        for (const p of paragraphs) {
            const text = p.textContent.trim();
            if (text.length > longestText.length && text.length < 10000) {
                longestText = text;
            }
        }

        return longestText || 'Unable to extract job description';
    },

    // Main extraction method
    extract() {
        try {
            const title = this.findJobTitle();
            const company = this.findCompany();
            const location = this.findLocation();
            const description = this.findDescription();

            return {
                success: true,
                site: 'generic',
                title,
                company,
                location,
                description,
                jobUrl: window.location.href,
                extractedAt: new Date().toISOString(),
                confidence: this.calculateConfidence(title, company, description)
            };
        } catch (error) {
            console.error('Generic extraction error:', error);
            return {
                success: false,
                site: 'generic',
                error: error.message
            };
        }
    },

    // Calculate confidence score for extraction quality
    calculateConfidence(title, company, description) {
        let score = 0;

        if (title && title.length > 5) score += 30;
        if (company && company.length > 1) score += 25;
        if (description && description.length > 200) score += 30;
        if (description && description.length > 500) score += 15;

        return Math.min(score, 100);
    }
};

export default GenericExtractor;
