// Greenhouse ATS Extractor
// Used by many tech companies (Stripe, Airbnb, Coinbase, etc.)

const GreenhouseExtractor = {
    name: 'Greenhouse',

    isJobPage() {
        return window.location.hostname.includes('greenhouse.io') ||
            window.location.hostname.includes('boards.greenhouse.io') ||
            document.querySelector('#app_form') !== null ||
            document.querySelector('[data-mapped="true"]') !== null;
    },

    extract() {
        try {
            // Greenhouse has consistent DOM structure
            const title = document.querySelector('.app-title, h1.heading')?.textContent.trim() ||
                document.querySelector('[class*="job-title"], [class*="JobTitle"]')?.textContent.trim() || '';

            const company = document.querySelector('.company-name, [class*="company"]')?.textContent.trim() ||
                this.extractCompanyFromUrl() || '';

            const location = document.querySelector('.location, [class*="location"]')?.textContent.trim() || '';

            const description = document.querySelector('#content, .content, [class*="description"]')?.textContent.trim() || '';

            return {
                success: true,
                site: 'greenhouse',
                title,
                company,
                location,
                description,
                hasDirectApply: !!document.querySelector('#application_form, #app_form, [data-mapped="true"]'),
                jobUrl: window.location.href,
                jobId: this.extractJobId()
            };
        } catch (error) {
            return { success: false, site: 'greenhouse', error: error.message };
        }
    },

    extractCompanyFromUrl() {
        // URL format: boards.greenhouse.io/companyname/jobs/123
        const match = window.location.pathname.match(/^\/([^\/]+)/);
        return match ? match[1].replace(/-/g, ' ') : '';
    },

    extractJobId() {
        const match = window.location.pathname.match(/jobs\/(\d+)/);
        return match ? match[1] : null;
    },

    async fillApplication(resume, coverLetter, userInfo) {
        try {
            const fillField = (selector, value) => {
                const el = document.querySelector(selector);
                if (el && value) {
                    el.value = value;
                    el.dispatchEvent(new Event('input', { bubbles: true }));
                    el.dispatchEvent(new Event('change', { bubbles: true }));
                }
            };

            // Standard Greenhouse fields
            fillField('#first_name, [name*="first_name"]', userInfo?.firstName);
            fillField('#last_name, [name*="last_name"]', userInfo?.lastName);
            fillField('#email, [name*="email"]', userInfo?.email);
            fillField('#phone, [name*="phone"]', userInfo?.phone);

            // Cover letter
            const coverLetterField = document.querySelector('textarea[name*="cover_letter"], #cover_letter');
            if (coverLetterField && coverLetter) {
                coverLetterField.value = coverLetter;
                coverLetterField.dispatchEvent(new Event('input', { bubbles: true }));
            }

            return { success: true, message: 'Form filled. Please review and submit.' };
        } catch (error) {
            return { success: false, error: error.message };
        }
    }
};

export default GreenhouseExtractor;
