// Indeed Job Extractor
// Extracts job information from Indeed job pages

const IndeedExtractor = {
    name: 'Indeed',

    // Check if current page is a job posting
    isJobPage() {
        const url = window.location.href;
        return url.includes('/viewjob') ||
            url.includes('/rc/clk') ||
            document.querySelector('[data-testid="jobsearch-ViewJobLayout"]') !== null;
    },

    // Extract job information from the page
    extract() {
        try {
            // Job title
            const titleSelectors = [
                '[data-testid="jobsearch-JobInfoHeader-title"]',
                '.jobsearch-JobInfoHeader-title',
                'h1.icl-u-xs-mb--xs',
                '.jobTitle',
                'h1[class*="JobInfo"]'
            ];

            let title = '';
            for (const selector of titleSelectors) {
                const el = document.querySelector(selector);
                if (el) {
                    title = el.textContent.trim();
                    break;
                }
            }

            // Company name
            const companySelectors = [
                '[data-testid="inlineHeader-companyName"]',
                '[data-testid="jobsearch-CompanyInfoContainer"] a',
                '.jobsearch-InlineCompanyRating a',
                '.icl-u-lg-mr--sm a',
                '[class*="CompanyName"]'
            ];

            let company = '';
            for (const selector of companySelectors) {
                const el = document.querySelector(selector);
                if (el) {
                    company = el.textContent.trim();
                    break;
                }
            }

            // Location
            const locationSelectors = [
                '[data-testid="job-location"]',
                '[data-testid="jobsearch-JobInfoHeader-companyLocation"]',
                '.jobsearch-JobInfoHeader-subtitle > div:last-child',
                '.icl-u-xs-mt--xs .icl-u-textColor--secondary'
            ];

            let location = '';
            for (const selector of locationSelectors) {
                const el = document.querySelector(selector);
                if (el) {
                    location = el.textContent.trim();
                    break;
                }
            }

            // Job description
            const descriptionSelectors = [
                '#jobDescriptionText',
                '.jobsearch-jobDescriptionText',
                '[data-testid="jobsearch-JobComponent-description"]',
                '.jobDescription'
            ];

            let description = '';
            for (const selector of descriptionSelectors) {
                const el = document.querySelector(selector);
                if (el) {
                    description = el.textContent.trim();
                    break;
                }
            }

            // Salary info (if available)
            const salarySelectors = [
                '[data-testid="jobsearch-JobInfoHeader-salary"]',
                '.jobsearch-JobMetadataHeader-item .attribute_snippet',
                '#salaryInfoAndJobType'
            ];

            let salary = '';
            for (const selector of salarySelectors) {
                const el = document.querySelector(selector);
                if (el) {
                    const text = el.textContent.trim();
                    if (text.includes('$') || text.toLowerCase().includes('year') || text.toLowerCase().includes('hour')) {
                        salary = text;
                        break;
                    }
                }
            }

            // Check for Apply button
            const applySelectors = [
                '#indeedApplyButton',
                '[data-testid="indeedApply-button"]',
                '.jobsearch-IndeedApplyButton-contentWrapper',
                'button[id*="apply"]'
            ];

            let hasDirectApply = false;
            let applyButton = null;
            for (const selector of applySelectors) {
                const el = document.querySelector(selector);
                if (el) {
                    hasDirectApply = true;
                    applyButton = el;
                    break;
                }
            }

            // Get job key from URL
            const jobKeyMatch = window.location.href.match(/jk=([a-f0-9]+)/i);
            const jobId = jobKeyMatch ? jobKeyMatch[1] : null;

            return {
                success: true,
                site: 'indeed',
                jobId,
                title,
                company,
                location,
                description,
                salary,
                hasDirectApply,
                applyButton,
                jobUrl: window.location.href,
                extractedAt: new Date().toISOString()
            };
        } catch (error) {
            console.error('Indeed extraction error:', error);
            return {
                success: false,
                site: 'indeed',
                error: error.message
            };
        }
    },

    // Fill Indeed application form
    async fillApplication(resume, coverLetter, userInfo) {
        try {
            // Click Apply button
            const applyButton = document.querySelector('#indeedApplyButton') ||
                document.querySelector('[data-testid="indeedApply-button"]');

            if (!applyButton) {
                return { success: false, error: 'Apply button not found' };
            }

            applyButton.click();

            // Wait for modal/form to appear
            await new Promise(resolve => setTimeout(resolve, 1500));

            // Fill common form fields
            const fillField = (selectors, value) => {
                for (const selector of selectors) {
                    const el = document.querySelector(selector);
                    if (el && value) {
                        el.value = value;
                        el.dispatchEvent(new Event('input', { bubbles: true }));
                        el.dispatchEvent(new Event('change', { bubbles: true }));
                        return true;
                    }
                }
                return false;
            };

            // Name fields
            if (userInfo?.firstName) {
                fillField(['input[name="firstName"]', 'input[id*="firstName"]'], userInfo.firstName);
            }
            if (userInfo?.lastName) {
                fillField(['input[name="lastName"]', 'input[id*="lastName"]'], userInfo.lastName);
            }
            if (userInfo?.email) {
                fillField(['input[name="email"]', 'input[type="email"]'], userInfo.email);
            }
            if (userInfo?.phone) {
                fillField(['input[name="phone"]', 'input[type="tel"]'], userInfo.phone);
            }

            // Cover letter textarea
            const coverLetterTextareas = document.querySelectorAll('textarea');
            for (const textarea of coverLetterTextareas) {
                const label = textarea.getAttribute('aria-label') ||
                    textarea.getAttribute('placeholder') ||
                    textarea.closest('label')?.textContent || '';

                if (label.toLowerCase().includes('cover') || label.toLowerCase().includes('message')) {
                    textarea.value = coverLetter;
                    textarea.dispatchEvent(new Event('input', { bubbles: true }));
                }
            }

            // Resume upload
            const fileInput = document.querySelector('input[type="file"][accept*="pdf"], input[type="file"][accept*="doc"]');
            if (fileInput && resume) {
                // Note: Direct file upload from extension requires special handling
                console.log('Resume upload field found - manual upload required');
            }

            return { success: true, message: 'Form filled. Please review and submit.' };
        } catch (error) {
            return { success: false, error: error.message };
        }
    }
};

export default IndeedExtractor;
