// Lever ATS Extractor
// Popular ATS used by Netflix, Twitch, Shopify, etc.

const LeverExtractor = {
    name: 'Lever',

    isJobPage() {
        return window.location.hostname.includes('lever.co') ||
            window.location.hostname.includes('jobs.lever.co') ||
            document.querySelector('.posting-headline') !== null;
    },

    extract() {
        try {
            const title = document.querySelector('.posting-headline h2, [data-qa="posting-name"]')?.textContent.trim() || '';

            const companyEl = document.querySelector('.posting-headline .sort-by-time, .main-header-logo img');
            const company = companyEl?.getAttribute('alt') ||
                document.querySelector('.posting-categories .location')?.previousElementSibling?.textContent.trim() ||
                this.extractCompanyFromUrl() || '';

            const location = document.querySelector('.posting-categories .location, [data-qa="posting-location"]')?.textContent.trim() || '';

            const description = document.querySelector('.posting-description, [data-qa="posting-description"]')?.textContent.trim() || '';

            const commitment = document.querySelector('.posting-categories .commitment')?.textContent.trim() || '';
            const team = document.querySelector('.posting-categories .team')?.textContent.trim() || '';

            return {
                success: true,
                site: 'lever',
                title,
                company,
                location,
                description,
                commitment,
                team,
                hasDirectApply: !!document.querySelector('.postings-btn-wrapper a, [data-qa="btn-apply"]'),
                jobUrl: window.location.href,
                jobId: this.extractJobId()
            };
        } catch (error) {
            return { success: false, site: 'lever', error: error.message };
        }
    },

    extractCompanyFromUrl() {
        // URL format: jobs.lever.co/companyname/job-id
        const match = window.location.pathname.match(/^\/([^\/]+)/);
        return match ? match[1].replace(/-/g, ' ') : '';
    },

    extractJobId() {
        const match = window.location.pathname.match(/\/([a-f0-9-]{36})/i);
        return match ? match[1] : null;
    },

    async fillApplication(resume, coverLetter, userInfo) {
        try {
            // Click apply button first
            const applyBtn = document.querySelector('.postings-btn-wrapper a, [data-qa="btn-apply"]');
            if (applyBtn) applyBtn.click();

            await new Promise(resolve => setTimeout(resolve, 1000));

            const fillField = (selector, value) => {
                const el = document.querySelector(selector);
                if (el && value) {
                    el.value = value;
                    el.dispatchEvent(new Event('input', { bubbles: true }));
                }
            };

            // Lever form fields
            fillField('input[name="name"]', `${userInfo?.firstName || ''} ${userInfo?.lastName || ''}`);
            fillField('input[name="email"]', userInfo?.email);
            fillField('input[name="phone"]', userInfo?.phone);
            fillField('input[name="org"], input[name="company"]', userInfo?.currentCompany);

            // Cover letter / additional info
            const textareas = document.querySelectorAll('textarea');
            textareas.forEach(ta => {
                if (coverLetter) {
                    ta.value = coverLetter;
                    ta.dispatchEvent(new Event('input', { bubbles: true }));
                }
            });

            return { success: true, message: 'Form filled. Please review and submit.' };
        } catch (error) {
            return { success: false, error: error.message };
        }
    }
};

export default LeverExtractor;
