// LinkedIn Job Extractor
// Extracts job information from LinkedIn job pages

const LinkedInExtractor = {
    name: 'LinkedIn',

    // Check if current page is a job posting
    isJobPage() {
        const url = window.location.href;
        return url.includes('/jobs/view/') ||
            url.includes('/jobs/search/') && document.querySelector('.job-details-jobs-unified-top-card__job-title');
    },

    // Extract job information from the page
    extract() {
        try {
            // Job title - try multiple selectors for different LinkedIn layouts
            const titleSelectors = [
                '.job-details-jobs-unified-top-card__job-title',
                '.jobs-unified-top-card__job-title',
                'h1.t-24',
                '.job-details h1',
                '[data-test-job-title]'
            ];

            let title = '';
            for (const selector of titleSelectors) {
                const el = document.querySelector(selector);
                if (el) {
                    title = el.textContent.trim();
                    break;
                }
            }

            // Company name
            const companySelectors = [
                '.job-details-jobs-unified-top-card__company-name',
                '.jobs-unified-top-card__company-name',
                '.job-details-jobs-unified-top-card__primary-description-container a',
                '[data-test-company-name]'
            ];

            let company = '';
            for (const selector of companySelectors) {
                const el = document.querySelector(selector);
                if (el) {
                    company = el.textContent.trim();
                    break;
                }
            }

            // Location
            const locationSelectors = [
                '.job-details-jobs-unified-top-card__bullet',
                '.jobs-unified-top-card__bullet',
                '.job-details-jobs-unified-top-card__primary-description-container .t-black--light',
                '[data-test-job-location]'
            ];

            let location = '';
            for (const selector of locationSelectors) {
                const el = document.querySelector(selector);
                if (el) {
                    location = el.textContent.trim();
                    break;
                }
            }

            // Job description
            const descriptionSelectors = [
                '.jobs-description-content__text',
                '.jobs-description__content',
                '#job-details',
                '.job-view-layout article'
            ];

            let description = '';
            for (const selector of descriptionSelectors) {
                const el = document.querySelector(selector);
                if (el) {
                    description = el.textContent.trim();
                    break;
                }
            }

            // Check for Easy Apply button
            const easyApplySelectors = [
                '[data-control-name="jobdetails_topcard_inapply"]',
                '.jobs-apply-button--top-card',
                'button[aria-label*="Easy Apply"]',
                '.jobs-s-apply button'
            ];

            let hasEasyApply = false;
            let easyApplyButton = null;
            for (const selector of easyApplySelectors) {
                const el = document.querySelector(selector);
                if (el) {
                    const text = el.textContent.toLowerCase();
                    if (text.includes('easy apply') || text.includes('apply')) {
                        hasEasyApply = true;
                        easyApplyButton = el;
                        break;
                    }
                }
            }

            // Get job ID from URL
            const jobIdMatch = window.location.href.match(/\/jobs\/view\/(\d+)/);
            const jobId = jobIdMatch ? jobIdMatch[1] : null;

            return {
                success: true,
                site: 'linkedin',
                jobId,
                title,
                company,
                location,
                description,
                hasEasyApply,
                easyApplyButton,
                jobUrl: window.location.href,
                extractedAt: new Date().toISOString()
            };
        } catch (error) {
            console.error('LinkedIn extraction error:', error);
            return {
                success: false,
                site: 'linkedin',
                error: error.message
            };
        }
    },

    // Fill Easy Apply form with generated content
    async fillEasyApply(resume, coverLetter) {
        try {
            // Click Easy Apply button
            const applyButton = document.querySelector('[data-control-name="jobdetails_topcard_inapply"]') ||
                document.querySelector('.jobs-apply-button--top-card');

            if (!applyButton) {
                return { success: false, error: 'Easy Apply button not found' };
            }

            applyButton.click();

            // Wait for modal to appear
            await new Promise(resolve => setTimeout(resolve, 1000));

            // Look for resume upload
            const fileInput = document.querySelector('input[type="file"]');
            if (fileInput && resume) {
                // Note: File upload requires creating a File object from the resume
                // This is complex and may require backend support
            }

            // Fill text areas (cover letter, additional questions)
            const textAreas = document.querySelectorAll('textarea');
            for (const textarea of textAreas) {
                const label = textarea.getAttribute('aria-label') ||
                    textarea.closest('label')?.textContent || '';

                if (label.toLowerCase().includes('cover') && coverLetter) {
                    textarea.value = coverLetter;
                    textarea.dispatchEvent(new Event('input', { bubbles: true }));
                }
            }

            return { success: true, message: 'Form filled. Please review and submit.' };
        } catch (error) {
            return { success: false, error: error.message };
        }
    }
};

export default LinkedInExtractor;
