// Workday ATS Extractor
// Enterprise ATS used by Fortune 500 companies

const WorkdayExtractor = {
    name: 'Workday',

    isJobPage() {
        return window.location.hostname.includes('myworkday') ||
            window.location.hostname.includes('workday.com') ||
            window.location.hostname.includes('wd1.myworkdayjobs.com') ||
            window.location.hostname.includes('wd3.myworkdayjobs.com') ||
            window.location.hostname.includes('wd5.myworkdayjobs.com') ||
            document.querySelector('[data-automation-id="jobPostingHeader"]') !== null;
    },

    extract() {
        try {
            // Workday uses data-automation-id attributes
            const title = document.querySelector('[data-automation-id="jobPostingHeader"], h2[data-automation-id]')?.textContent.trim() ||
                document.querySelector('.css-1q2dra3, [class*="jobTitle"]')?.textContent.trim() || '';

            const company = this.extractCompanyFromUrl() ||
                document.querySelector('[data-automation-id="companyName"]')?.textContent.trim() || '';

            const locationEl = document.querySelector('[data-automation-id="locations"], [data-automation-id="jobPostingLocation"]');
            const location = locationEl?.textContent.trim() || '';

            const description = document.querySelector('[data-automation-id="jobPostingDescription"], .job-description')?.textContent.trim() || '';

            // Posted date
            const posted = document.querySelector('[data-automation-id="postedOn"]')?.textContent.trim() || '';

            return {
                success: true,
                site: 'workday',
                title,
                company,
                location,
                description,
                posted,
                hasDirectApply: !!document.querySelector('[data-automation-id="applyButton"], button[aria-label*="Apply"]'),
                jobUrl: window.location.href,
                jobId: this.extractJobId()
            };
        } catch (error) {
            return { success: false, site: 'workday', error: error.message };
        }
    },

    extractCompanyFromUrl() {
        // URL format: company.wd1.myworkdayjobs.com
        const match = window.location.hostname.match(/^([^.]+)\.wd/);
        return match ? match[1].replace(/-/g, ' ') : '';
    },

    extractJobId() {
        const match = window.location.pathname.match(/job\/([^\/]+)/);
        return match ? match[1] : null;
    },

    async fillApplication(resume, coverLetter, userInfo) {
        try {
            // Click apply button
            const applyBtn = document.querySelector('[data-automation-id="applyButton"]');
            if (applyBtn) applyBtn.click();

            await new Promise(resolve => setTimeout(resolve, 2000));

            // Workday forms are complex - fill basic fields
            const inputs = document.querySelectorAll('input[type="text"], input[type="email"], input[type="tel"]');
            inputs.forEach(input => {
                const label = input.getAttribute('aria-label') || input.getAttribute('placeholder') || '';
                const labelLower = label.toLowerCase();

                if (labelLower.includes('first') && userInfo?.firstName) {
                    input.value = userInfo.firstName;
                } else if (labelLower.includes('last') && userInfo?.lastName) {
                    input.value = userInfo.lastName;
                } else if (labelLower.includes('email') && userInfo?.email) {
                    input.value = userInfo.email;
                } else if (labelLower.includes('phone') && userInfo?.phone) {
                    input.value = userInfo.phone;
                }
                input.dispatchEvent(new Event('input', { bubbles: true }));
                input.dispatchEvent(new Event('change', { bubbles: true }));
            });

            return { success: true, message: 'Basic fields filled. Workday forms are complex - please complete remaining fields.' };
        } catch (error) {
            return { success: false, error: error.message };
        }
    }
};

export default WorkdayExtractor;
