// Chrome Storage Wrapper
// Provides easy access to extension storage

class Storage {
    // Get value from storage
    static async get(key, defaultValue = null) {
        const result = await chrome.storage.local.get(key);
        return result[key] !== undefined ? result[key] : defaultValue;
    }

    // Set value in storage
    static async set(key, value) {
        await chrome.storage.local.set({ [key]: value });
    }

    // Remove value from storage
    static async remove(key) {
        await chrome.storage.local.remove(key);
    }

    // Get multiple values
    static async getMultiple(keys) {
        return await chrome.storage.local.get(keys);
    }

    // Application history management
    static async getApplicationHistory() {
        return await this.get('applicationHistory', []);
    }

    static async addApplication(application) {
        const history = await this.getApplicationHistory();
        const newApp = {
            id: Date.now().toString(),
            ...application,
            appliedAt: new Date().toISOString(),
            status: 'applied'
        };
        history.unshift(newApp);

        // Keep only last 500 applications
        if (history.length > 500) {
            history.pop();
        }

        await this.set('applicationHistory', history);
        return newApp;
    }

    static async updateApplicationStatus(id, status) {
        const history = await this.getApplicationHistory();
        const index = history.findIndex(app => app.id === id);
        if (index !== -1) {
            history[index].status = status;
            history[index].updatedAt = new Date().toISOString();
            await this.set('applicationHistory', history);
        }
    }

    // Check if already applied to a job
    static async hasApplied(jobUrl) {
        const history = await this.getApplicationHistory();
        return history.some(app => app.jobUrl === jobUrl);
    }

    // Get today's application count
    static async getTodayCount() {
        const history = await this.getApplicationHistory();
        const today = new Date().toDateString();
        return history.filter(app =>
            new Date(app.appliedAt).toDateString() === today
        ).length;
    }

    // Settings management
    static async getSettings() {
        return await this.get('settings', {
            autoFillOnly: true,  // Don't auto-submit, just fill
            dailyLimit: 20,
            showNotifications: true,
            preferredStyle: 'professional-modern',
            preferredFormat: 'pdf',  // Default to PDF output
            confirmBeforeApply: true
        });
    }

    static async updateSettings(newSettings) {
        const current = await this.getSettings();
        await this.set('settings', { ...current, ...newSettings });
    }

    // Blacklist/whitelist
    static async getBlacklist() {
        return await this.get('companyBlacklist', []);
    }

    static async addToBlacklist(company) {
        const list = await this.getBlacklist();
        if (!list.includes(company.toLowerCase())) {
            list.push(company.toLowerCase());
            await this.set('companyBlacklist', list);
        }
    }

    static async isBlacklisted(company) {
        const list = await this.getBlacklist();
        return list.includes(company.toLowerCase());
    }
}

export default Storage;
