// exploit.team AutoJob Options Script

// Settings elements
const autoFillOnly = document.getElementById('autoFillOnly');
const confirmBeforeApply = document.getElementById('confirmBeforeApply');
const dailyLimit = document.getElementById('dailyLimit');
const preferredStyle = document.getElementById('preferredStyle');
const preferredFormat = document.getElementById('preferredFormat');
const showNotifications = document.getElementById('showNotifications');
const blacklistInput = document.getElementById('blacklistInput');
const addBlacklistBtn = document.getElementById('addBlacklist');
const blacklistItems = document.getElementById('blacklistItems');
const accountInfo = document.getElementById('accountInfo');
const logoutBtn = document.getElementById('logoutBtn');
const exportDataBtn = document.getElementById('exportData');
const clearHistoryBtn = document.getElementById('clearHistory');

// Profile elements
const profileFirstName = document.getElementById('profileFirstName');
const profileLastName = document.getElementById('profileLastName');
const profileEmail = document.getElementById('profileEmail');
const profilePhone = document.getElementById('profilePhone');
const profileCity = document.getElementById('profileCity');
const profileState = document.getElementById('profileState');
const profileAddress1 = document.getElementById('profileAddress1');
const profileAddress2 = document.getElementById('profileAddress2');
const profileZip = document.getElementById('profileZip');
const profileCountry = document.getElementById('profileCountry');
const profileYearsExp = document.getElementById('profileYearsExp');
const profilePortfolio = document.getElementById('profilePortfolio');
const profileLinkedIn = document.getElementById('profileLinkedIn');
const profileWorkAuth = document.getElementById('profileWorkAuth');
const profileSponsorship = document.getElementById('profileSponsorship');
const profileResume = document.getElementById('profileResume');
const profileGenerationNotes = document.getElementById('profileGenerationNotes');
const saveProfileBtn = document.getElementById('saveProfile');

// Initialize
async function init() {
    await loadSettings();
    await loadProfile();
    await loadBlacklist();
    await loadAccountInfo();
    setupListeners();
}

// Load user profile
async function loadProfile() {
    const response = await chrome.runtime.sendMessage({ action: 'getUserProfile' });
    if (response.success && response.profile) {
        const p = response.profile;
        if (profileFirstName) profileFirstName.value = p.firstName || '';
        if (profileLastName) profileLastName.value = p.lastName || '';
        if (profileEmail) profileEmail.value = p.email || '';
        if (profilePhone) profilePhone.value = p.phone || '';
        if (profileCity) profileCity.value = p.city || '';
        if (profileState) profileState.value = p.state || '';
        if (profileAddress1) profileAddress1.value = p.address1 || '';
        if (profileAddress2) profileAddress2.value = p.address2 || '';
        if (profileZip) profileZip.value = p.zip || '';
        if (profileCountry) profileCountry.value = p.country || '';
        if (profileYearsExp) profileYearsExp.value = p.yearsExperience || '';
        if (profilePortfolio) profilePortfolio.value = p.portfolioUrl || '';
        if (profileLinkedIn) profileLinkedIn.value = p.linkedinUrl || '';
        if (profileWorkAuth) profileWorkAuth.checked = p.workAuthorized !== false;
        if (profileSponsorship) profileSponsorship.checked = p.needsSponsorship === true;
        if (profileResume) profileResume.value = p.resumeText || '';
        if (profileGenerationNotes) profileGenerationNotes.value = p.generationNotes || '';
    }
}

// Save user profile
async function saveProfile() {
    const profile = {
        firstName: profileFirstName?.value?.trim() || '',
        lastName: profileLastName?.value?.trim() || '',
        email: profileEmail?.value?.trim() || '',
        phone: profilePhone?.value?.trim() || '',
        city: profileCity?.value?.trim() || '',
        state: profileState?.value?.trim() || '',
        address1: profileAddress1?.value?.trim() || '',
        address2: profileAddress2?.value?.trim() || '',
        zip: profileZip?.value?.trim() || '',
        country: profileCountry?.value?.trim() || '',
        yearsExperience: parseInt(profileYearsExp?.value) || 0,
        portfolioUrl: profilePortfolio?.value?.trim() || '',
        linkedinUrl: profileLinkedIn?.value?.trim() || '',
        workAuthorized: profileWorkAuth?.checked ?? true,
        needsSponsorship: profileSponsorship?.checked ?? false,
        resumeText: profileResume?.value || '',
        generationNotes: profileGenerationNotes?.value || ''
    };

    await chrome.runtime.sendMessage({ action: 'saveUserProfile', data: profile });

    // Show success
    const btn = saveProfileBtn;
    const originalText = btn.textContent;
    btn.textContent = '✅ Saved!';
    btn.style.background = '#16a34a';
    setTimeout(() => {
        btn.textContent = originalText;
        btn.style.background = '#22c55e';
    }, 2000);
}

// Load settings
async function loadSettings() {
    const response = await chrome.runtime.sendMessage({ action: 'getSettings' });
    if (response.success) {
        const settings = response.settings;
        autoFillOnly.checked = settings.autoFillOnly ?? true;
        confirmBeforeApply.checked = settings.confirmBeforeApply ?? true;
        dailyLimit.value = settings.dailyLimit ?? 20;
        preferredStyle.value = settings.preferredStyle ?? 'professional-modern';
        preferredFormat.value = settings.preferredFormat ?? 'pdf';
        showNotifications.checked = settings.showNotifications ?? true;
    }
}

// Save settings
async function saveSettings() {
    const settings = {
        autoFillOnly: autoFillOnly.checked,
        confirmBeforeApply: confirmBeforeApply.checked,
        dailyLimit: parseInt(dailyLimit.value) || 20,
        preferredStyle: preferredStyle.value,
        preferredFormat: preferredFormat.value,
        showNotifications: showNotifications.checked
    };

    await chrome.runtime.sendMessage({ action: 'updateSettings', data: settings });
}

// Load blacklist
async function loadBlacklist() {
    const result = await chrome.storage.local.get('companyBlacklist');
    const blacklist = result.companyBlacklist || [];
    renderBlacklist(blacklist);
}

// Render blacklist
function renderBlacklist(blacklist) {
    if (blacklist.length === 0) {
        blacklistItems.innerHTML = '<p style="color: #6b7280; font-size: 13px;">No companies blacklisted</p>';
        return;
    }

    blacklistItems.innerHTML = blacklist.map(company => `
    <span class="blacklist-tag">
      ${escapeHtml(company)}
      <button onclick="removeFromBlacklist('${escapeHtml(company)}')" title="Remove">×</button>
    </span>
  `).join('');
}

// Add to blacklist
async function addToBlacklist() {
    const company = blacklistInput.value.trim().toLowerCase();
    if (!company) return;

    const result = await chrome.storage.local.get('companyBlacklist');
    const blacklist = result.companyBlacklist || [];

    if (!blacklist.includes(company)) {
        blacklist.push(company);
        await chrome.storage.local.set({ companyBlacklist: blacklist });
        renderBlacklist(blacklist);
    }

    blacklistInput.value = '';
}

// Remove from blacklist
window.removeFromBlacklist = async function (company) {
    const result = await chrome.storage.local.get('companyBlacklist');
    let blacklist = result.companyBlacklist || [];
    blacklist = blacklist.filter(c => c !== company);
    await chrome.storage.local.set({ companyBlacklist: blacklist });
    renderBlacklist(blacklist);
};

// Load account info
async function loadAccountInfo() {
    try {
        // Try to get auth status from background script
        const response = await chrome.runtime.sendMessage({ action: 'checkAuth' });

        if (response && response.isAuthenticated) {
            const user = response.user;
            accountInfo.innerHTML = `
          <p><strong>Username:</strong> ${escapeHtml(user?.username || user?.id || 'Unknown')}</p>
          <p><strong>Status:</strong> <span style="color: #28ca41;">Connected</span></p>
        `;
            return;
        }
    } catch (e) {
        console.log('checkAuth failed, trying direct storage:', e);
    }

    // Fallback: check storage directly
    try {
        const stored = await chrome.storage.local.get(['authToken', 'user']);
        if (stored.authToken && stored.user) {
            accountInfo.innerHTML = `
          <p><strong>Username:</strong> ${escapeHtml(stored.user?.username || stored.user?.id || 'Unknown')}</p>
          <p><strong>Status:</strong> <span style="color: #28ca41;">Connected</span></p>
        `;
            return;
        }
    } catch (e) {
        console.log('Direct storage check failed:', e);
    }

    // Not authenticated
    accountInfo.innerHTML = `
      <p><strong>Status:</strong> <span style="color: #ef4444;">Not logged in</span></p>
      <p>Login via the extension popup to sync your account.</p>
    `;
}

// Export data
async function exportData() {
    const response = await chrome.runtime.sendMessage({ action: 'getApplicationHistory' });

    if (response.success) {
        const data = JSON.stringify(response.history, null, 2);
        const blob = new Blob([data], { type: 'application/json' });
        const url = URL.createObjectURL(blob);

        const a = document.createElement('a');
        a.href = url;
        a.download = `jobgen-history-${new Date().toISOString().split('T')[0]}.json`;
        a.click();

        URL.revokeObjectURL(url);
    }
}

// Clear history
async function clearHistory() {
    if (confirm('Are you sure you want to clear all application history? This cannot be undone.')) {
        await chrome.storage.local.remove('applicationHistory');
        alert('Application history cleared.');
    }
}

// Logout
async function logout() {
    if (confirm('Are you sure you want to logout?')) {
        await chrome.runtime.sendMessage({ action: 'logout' });
        loadAccountInfo();
    }
}

// Setup event listeners
function setupListeners() {
    // Settings auto-save
    [autoFillOnly, confirmBeforeApply, showNotifications].forEach(el => {
        el.addEventListener('change', saveSettings);
    });

    [dailyLimit, preferredStyle, preferredFormat].forEach(el => {
        el.addEventListener('change', saveSettings);
    });

    // Blacklist
    addBlacklistBtn.addEventListener('click', addToBlacklist);
    blacklistInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') addToBlacklist();
    });

    // Account actions
    logoutBtn.addEventListener('click', logout);

    // Data management
    exportDataBtn.addEventListener('click', exportData);
    clearHistoryBtn.addEventListener('click', clearHistory);

    // Profile save
    saveProfileBtn?.addEventListener('click', saveProfile);
}

// Helper
function escapeHtml(str) {
    if (!str) return '';
    return str.replace(/[&<>"']/g, char => ({
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#39;'
    })[char]);
}

// Initialize
init();
