// Eric's AutoApply Popup Script

// Update check URL
const UPDATE_CHECK_URL = 'https://job.exploit.bot/extension-update.json';

// Elements
const authSection = document.getElementById('auth-section');
const dashboardSection = document.getElementById('dashboard-section');
const loginForm = document.getElementById('login-form');
const authError = document.getElementById('auth-error');
const userEmail = document.getElementById('user-email');
const logoutBtn = document.getElementById('logout-btn');
const settingsBtn = document.getElementById('settings-btn');
const todayCount = document.getElementById('today-count');
const weekCount = document.getElementById('week-count');
const totalCount = document.getElementById('total-count');
const statusDot = document.getElementById('status-dot');
const statusText = document.getElementById('status-text');
const generateBtn = document.getElementById('generate-btn');
const recentList = document.getElementById('recent-list');
const updateBanner = document.getElementById('update-banner');
const updateBtn = document.getElementById('update-btn');
const forceBtn = document.getElementById('force-btn');
const fillNowBtn = document.getElementById('fill-now-btn');
const autoFillEnabled = document.getElementById('auto-fill-enabled');

// Initialize popup
async function init() {
    try {
        // Check authentication status
        const response = await chrome.runtime.sendMessage({ action: 'checkAuth' });

        if (response && response.isAuthenticated) {
            showDashboard(response.user);
            loadStats();
            loadRecentApplications();
            checkCurrentPage();
            loadAutofillSettings();
        } else {
            showAuth();
        }
    } catch (error) {
        console.error('Popup init error:', error);
        showAuth(); // Show login on any error
    }
}

// Show auth section
function showAuth() {
    authSection.style.display = 'block';
    dashboardSection.style.display = 'none';
}

// Show dashboard section
function showDashboard(user) {
    authSection.style.display = 'none';
    dashboardSection.style.display = 'block';
    userEmail.textContent = user?.username || user?.email || 'User';
}

// Handle login
loginForm.addEventListener('submit', async (e) => {
    e.preventDefault();

    const username = document.getElementById('username').value;
    const password = document.getElementById('password').value;

    const response = await chrome.runtime.sendMessage({
        action: 'login',
        data: { username, password }
    });

    if (response.success) {
        showDashboard(response.user);
        loadStats();
        loadRecentApplications();
        checkCurrentPage();
    } else {
        authError.textContent = response.error || 'Login failed';
        authError.style.display = 'block';
    }
});

// Handle logout
logoutBtn.addEventListener('click', async () => {
    await chrome.runtime.sendMessage({ action: 'logout' });
    showAuth();
});

// Open settings
settingsBtn.addEventListener('click', () => {
    chrome.runtime.openOptionsPage();
});

// Load stats
async function loadStats() {
    const todayResponse = await chrome.runtime.sendMessage({ action: 'getTodayCount' });
    const historyResponse = await chrome.runtime.sendMessage({ action: 'getApplicationHistory' });

    if (todayResponse.success) {
        todayCount.textContent = todayResponse.count;
    }

    if (historyResponse.success) {
        const history = historyResponse.history;
        totalCount.textContent = history.length;

        // Calculate week count
        const oneWeekAgo = new Date();
        oneWeekAgo.setDate(oneWeekAgo.getDate() - 7);
        const weekApps = history.filter(app => new Date(app.appliedAt) > oneWeekAgo);
        weekCount.textContent = weekApps.length;
    }
}

// Load recent applications
async function loadRecentApplications() {
    const response = await chrome.runtime.sendMessage({ action: 'getApplicationHistory' });

    if (!response.success || response.history.length === 0) {
        recentList.innerHTML = '<p class="empty-state">No applications yet</p>';
        return;
    }

    const recentApps = response.history.slice(0, 5);

    recentList.innerHTML = recentApps.map(app => `
    <div class="recent-item">
      <div class="recent-icon">${getSiteIcon(app.site)}</div>
      <div class="recent-info">
        <div class="recent-title">${escapeHtml(app.title)}</div>
        <div class="recent-company">${escapeHtml(app.company)}</div>
      </div>
      <div class="recent-time">${formatTime(app.appliedAt)}</div>
    </div>
  `).join('');
}

// Check if current page is a job page
async function checkCurrentPage() {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

        if (!tab || !tab.url) {
            updatePageStatus(false, 'No page detected');
            return;
        }

        const url = tab.url;
        const isJobSite = ['linkedin.com', 'indeed.com', 'glassdoor.com', 'ziprecruiter.com']
            .some(site => url.includes(site));

        if (isJobSite) {
            // Try to get job data from content script
            try {
                const response = await chrome.tabs.sendMessage(tab.id, { action: 'getJobData' });
                if (response && response.title) {
                    updatePageStatus(true, `${response.title.slice(0, 30)}...`);
                    generateBtn.style.display = 'block';
                } else {
                    updatePageStatus(false, 'Navigate to a job page');
                }
            } catch {
                updatePageStatus(false, 'Navigate to a job page');
            }
        } else {
            updatePageStatus(false, 'Not on a job site');
        }
    } catch (error) {
        updatePageStatus(false, 'Unable to check page');
    }
}

// Update page status indicator
function updatePageStatus(isActive, text) {
    statusDot.className = 'status-dot ' + (isActive ? 'active' : 'inactive');
    statusText.textContent = text;
}

// Generate button click - opens job in current tab and triggers generation
generateBtn.addEventListener('click', async () => {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tab) {
        chrome.tabs.sendMessage(tab.id, { action: 'openQuickApply' });
        window.close();
    }
});

// Force detect button - manual trigger for any page
if (forceBtn) {
    forceBtn.addEventListener('click', async () => {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!tab) return;

        forceBtn.textContent = '⏳ Detecting...';
        forceBtn.disabled = true;

        try {
            // Try to send message to existing content script
            const response = await chrome.tabs.sendMessage(tab.id, { action: 'forceDetect' });
            if (response && response.success) {
                updatePageStatus(true, `Detected: ${response.title?.slice(0, 25) || 'Job Page'}...`);
                generateBtn.style.display = 'block';
                forceBtn.textContent = '✅ Detected!';
            } else {
                // Inject content script if not present
                await chrome.scripting.executeScript({
                    target: { tabId: tab.id },
                    files: ['content/detector.js']
                });

                // Wait a bit then try again
                setTimeout(async () => {
                    try {
                        const retryResponse = await chrome.tabs.sendMessage(tab.id, { action: 'forceDetect' });
                        if (retryResponse && retryResponse.success) {
                            updatePageStatus(true, `Detected: ${retryResponse.title?.slice(0, 25) || 'Job Page'}...`);
                            generateBtn.style.display = 'block';
                            forceBtn.textContent = '✅ Detected!';
                        } else {
                            forceBtn.textContent = '❌ Not a job page';
                        }
                    } catch {
                        forceBtn.textContent = '❌ Detection failed';
                    }
                    setTimeout(() => {
                        forceBtn.textContent = '🔧 Force Detect This Page';
                        forceBtn.disabled = false;
                    }, 2000);
                }, 1000);
                return;
            }
        } catch (error) {
            console.log('[Eric\'s AutoApply] Force detect error:', error);
            forceBtn.textContent = '❌ Error - try reloading page';
        }

        setTimeout(() => {
            forceBtn.textContent = '🔧 Force Detect This Page';
            forceBtn.disabled = false;
        }, 2000);
    });
}

// Helper functions
function getSiteIcon(site) {
    const icons = {
        linkedin: '💼',
        indeed: '🔍',
        glassdoor: '🏢',
        ziprecruiter: '📋',
        generic: '📄'
    };
    return icons[site] || '📄';
}

function formatTime(dateString) {
    const date = new Date(dateString);
    const now = new Date();
    const diff = now - date;

    const minutes = Math.floor(diff / 60000);
    const hours = Math.floor(diff / 3600000);
    const days = Math.floor(diff / 86400000);

    if (minutes < 1) return 'now';
    if (minutes < 60) return `${minutes}m`;
    if (hours < 24) return `${hours}h`;
    return `${days}d`;
}

function escapeHtml(str) {
    if (!str) return '';
    return str.replace(/[&<>"']/g, char => ({
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#39;'
    })[char]);
}

// Check for extension updates
async function checkForUpdates() {
    try {
        const response = await fetch(UPDATE_CHECK_URL + '?t=' + Date.now());
        if (!response.ok) return;

        const updateInfo = await response.json();
        const manifest = chrome.runtime.getManifest();
        const currentVersion = manifest.version;
        const latestVersion = updateInfo.version;

        console.log(`[Eric's AutoApply] Current: ${currentVersion}, Latest: ${latestVersion}`);

        // Compare versions
        if (isNewerVersion(latestVersion, currentVersion)) {
            if (updateBanner) {
                updateBanner.style.display = 'flex';
                updateBanner.setAttribute('data-url', updateInfo.download_url);
                updateBanner.setAttribute('data-changelog', updateInfo.changelog || '');
            }
        }
    } catch (error) {
        console.log('[Eric\'s AutoApply] Update check failed:', error);
    }
}

// Compare version strings (returns true if v1 > v2)
function isNewerVersion(v1, v2) {
    const parts1 = v1.split('.').map(Number);
    const parts2 = v2.split('.').map(Number);

    for (let i = 0; i < Math.max(parts1.length, parts2.length); i++) {
        const p1 = parts1[i] || 0;
        const p2 = parts2[i] || 0;
        if (p1 > p2) return true;
        if (p1 < p2) return false;
    }
    return false;
}

// Update button handler
if (updateBtn) {
    updateBtn.addEventListener('click', () => {
        const url = updateBanner?.getAttribute('data-url') || 'https://job.exploit.bot/jobgen-pro-extension.zip';
        chrome.tabs.create({ url: url });
    });
}

// Load autofill settings
async function loadAutofillSettings() {
    try {
        const response = await chrome.runtime.sendMessage({ action: 'getSettings' });
        if (response.success && autoFillEnabled) {
            autoFillEnabled.checked = response.settings.autoFillEnabled || false;
        }
    } catch (e) {
        console.log('Failed to load autofill settings:', e);
    }
}

// Save autofill setting on toggle
if (autoFillEnabled) {
    autoFillEnabled.addEventListener('change', async () => {
        try {
            await chrome.runtime.sendMessage({
                action: 'updateSettings',
                data: { autoFillEnabled: autoFillEnabled.checked }
            });
        } catch (e) {
            console.log('Failed to save autofill setting:', e);
        }
    });
}

// Fill This Page Now button handler
if (fillNowBtn) {
    fillNowBtn.addEventListener('click', async () => {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!tab) return;

        fillNowBtn.textContent = '⏳ Filling...';
        fillNowBtn.disabled = true;

        try {
            const response = await chrome.tabs.sendMessage(tab.id, { action: 'autofillPage' });
            
            if (response && response.success) {
                fillNowBtn.textContent = `✅ Filled ${response.filled} field${response.filled !== 1 ? 's' : ''}!`;
            } else {
                fillNowBtn.textContent = response?.error || '❌ No fields found';
            }
        } catch (e) {
            console.log('[Eric\'s AutoApply] Fill page error:', e);
            // Try injecting content scripts first
            try {
                await chrome.scripting.executeScript({
                    target: { tabId: tab.id },
                    files: ['content/page-analyzer.js', 'content/smart-autofill.js', 'content/detector.js']
                });
                
                // Retry after injection
                setTimeout(async () => {
                    try {
                        const retryResponse = await chrome.tabs.sendMessage(tab.id, { action: 'autofillPage' });
                        if (retryResponse && retryResponse.success) {
                            fillNowBtn.textContent = `✅ Filled ${retryResponse.filled} field${retryResponse.filled !== 1 ? 's' : ''}!`;
                        } else {
                            fillNowBtn.textContent = retryResponse?.error || '❌ No fields found';
                        }
                    } catch {
                        fillNowBtn.textContent = '❌ Failed - reload page';
                    }
                }, 500);
                return;
            } catch {
                fillNowBtn.textContent = '❌ Failed to inject';
            }
        }

        setTimeout(() => {
            fillNowBtn.textContent = '🖊️ Fill This Page Now';
            fillNowBtn.disabled = false;
        }, 2000);
    });
}

// Initialize on load
init();
checkForUpdates();

